<?php declare(strict_types = 1);

namespace EshopOrders\Model\Utils;

use Core\Model\Dao\Site;
use Core\Model\Templating\Filters\Price;
use Currency\Model\Config;
use Currency\Model\Exchange;
use EshopCatalog\FrontModule\Model\Dao\Seller;
use EshopOrders\Model\Entities\IDiscount;
use EshopOrders\Model\Entities\OrderDiscount;
use EshopOrders\Model\Entities\OrderItem;
use EshopOrders\Model\EshopOrdersConfig;
use EshopSales\Model\Entities\OrderSale;

class Helpers
{
	protected Price    $priceFilter;
	protected Exchange $exchange;

	public function __construct(
		Price    $priceFilter,
		Exchange $exchange
	)
	{
		$this->priceFilter = $priceFilter;
		$this->exchange    = $exchange;
	}

	public static function replaceTimeWildcards(string $input, array $timeWildcards): string
	{
		foreach ($timeWildcards as $item) {
			switch ($item) {
				case 'YYYY':
					$input = str_replace('YYYY', date('Y'), $input);
					break;
				case 'YY':
					$input = str_replace('YY', date('y'), $input);
					break;
				case 'MM':
					$input = str_replace('MM', date('m'), $input);
					break;
			}
		}

		return $input;
	}

	public function formatDiscount(IDiscount $discount): string
	{
		if ($discount instanceof OrderSale) {
			if ($discount->getValue()
				&& in_array($discount->getType(), [OrderDiscount::TYPE_FIX, OrderDiscount::TYPE_DELIVERY_PRICE])) {
				$value = $this->exchange->change((float) $discount->getValue());
			} else {
				$value = $discount->getValue();
			}
		} else {
			$value = $discount->getValue();
		}

		if (Config::load('roundOrderSales')) {
			$value = round($value);
		}

		if ($discount->getType() === OrderDiscount::TYPE_FIX) {
			return $this->priceFilter->format($value);
		}

		// type percent
		if ($discount->getType() === OrderDiscount::TYPE_PERCENT) {
			return sprintf('%s %s', $value, '%');
		}

		return (string) $value;
	}

	/**
	 * @param OrderItem[] $orderItems
	 */
	public static function isCorrectiveByOrderItems(array $orderItems): bool
	{
		foreach ($orderItems as $orderItem) {
			if ($orderItem->getPrice() < 0) {
				return true;
			}
		}

		return false;
	}

	/**
	 * @param Site|\Core\Model\Entities\Site $site
	 */
	public static function getSellerEmail(?Seller $seller, $site, ?string $lang = null): string
	{
		if ($seller && $seller->sendSellerEmailType === \EshopCatalog\Model\Entities\Seller::EMAIL_TYPE_SELLER) {
			$val = EshopOrdersConfig::load("orderForm.eshopSellerOtherEmail.{$site->getIdent()}_{$lang}.email", null)
				?: EshopOrdersConfig::load("orderForm.eshopSellerOtherEmail.{$site->getIdent()}.email", null);

			return $val ?: $seller->email;
		}

		return $site->email;
	}

	public static function getSellerName(?Seller $seller, string $siteIdent, ?string $lang = null): string
	{
		$val = EshopOrdersConfig::load("orderForm.eshopSellerOtherEmail.{$siteIdent}_{$lang}.name", null)
			?: EshopOrdersConfig::load("orderForm.eshopSellerOtherEmail.{$siteIdent}.name", null);

		return $val ?: ($seller->name ?? '');
	}
}
