<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Customer;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\Model\Config;
use EshopOrders\AdminModule\Model\GroupsCustomers;
use EshopOrders\Model\Entities\GroupCustomers;
use EshopOrders\Model\Entities\GroupParam;
use EshopOrders\Model\EshopOrdersConfig;
use Nette\InvalidArgumentException;
use Nette\Utils\ArrayHash;

class GroupCustomersForm extends BaseControl
{
	/** @var GroupCustomers */
	public $groupCustomers;

	/** @var GroupsCustomers */
	protected $groupsCustomersService;

	public function __construct(GroupsCustomers $groupsCustomers)
	{
		$this->groupsCustomersService = $groupsCustomers;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();

		$form->addText('short', 'eshopOrders.groupCustomersForm.short')->setMaxLength(60);
		$form->addText('name', 'eshopOrders.groupCustomersForm.name')->setRequired()->setMaxLength(255);
		$form->addBool('isPublished', 'eshopOrders.groupCustomersForm.isPublished')->setDefaultValue(0);

		if (Config::load('enablePriceLevels', false))
			$form->addText('productsSale', 'eshopOrders.groupCustomer.productsSale')
				->setHtmlType('number')
				->setHtmlAttribute('min', 0)
				->setHtmlAttribute('step', .01)
				->setDescription('eshopOrders.groupCustomer.productsSaleDescription')
				->setDefaultValue(0);

		$paramsContainer = $form->addContainer('params');
		if (EshopOrdersConfig::load('customerGroup.enableRequireIco'))
			$paramsContainer->addBool('requireIco', 'eshopOrders.groupCustomer.requireIco')
				->setDefaultValue(0);

		$form->addSaveCancelControl('saveControl');

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			$groupCustomers = $this->groupCustomers ?: new GroupCustomers($values->short, $values->name);

			$groupCustomers->short       = $values->short;
			$groupCustomers->name        = $values->name;
			$groupCustomers->isPublished = $values->isPublished;
			$groupCustomers->setProductsSale((float) $values->productsSale);

			/** @var GroupParam[] $params */
			$params = $groupCustomers->getParams()->toArray();
			bdump($params);

			foreach ($values->params as $k => $v) {
				if (isset($params[$k])) {
					$params[$k]->value = (string) $v;
					$this->em->persist($params[$k]);
				} else {
					$param        = new GroupParam($groupCustomers, $k);
					$param->value = (string) $v;
					$this->em->persist($param);
				}
			}

			$this->em->persist($groupCustomers);
			$this->em->flush();
			$form->addCustomData('groupCustomersId', $groupCustomers->getId());
			$this->getPresenter()->flashMessageSuccess('eshopOrders.groupCustomersForm.groupCustomersSaved');
		} catch (\Exception $e) {
			$form->addError($e->getMessage());
			$this->redrawControl('form');
		}

		return false;
	}

	public function setGroupCustomers($id): void
	{
		$this->groupCustomers = $this->em->getRepository(GroupCustomers::class)->find($id);

		if (!$this->groupCustomers)
			throw new InvalidArgumentException();

		$d = [
			'short'       => $this->groupCustomers->short,
			'name'        => $this->groupCustomers->name,
			'isPublished' => $this->groupCustomers->isPublished,
		];

		if (Config::load('enablePriceLevels', false))
			$d['productsSale'] = $this->groupCustomers->getProductsSale();

		foreach ($this->groupCustomers->getParams() as $k => $v)
			$d['params'][$k] = $v->value;

		$this['form']->setDefaults($d);
	}
}
