<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Order;

use Core\Model\Sites;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductTexts;
use EshopOrders\AdminModule\Model\Event\OrderItemEvent;
use Nette\Application\UI\Form;
use Nette\Utils\ArrayHash;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderItem;
use EshopOrders\Model\Orders;
use EshopOrders\Model\OrderItems;
use EshopCatalog\AdminModule\Model\Products;
use EshopCatalog\FrontModule\Model\Products as FrontProducts;

class OrderItemAddForm extends BaseControl
{
	public ?Order $order;

	protected Orders $ordersService;

	protected OrderItems $orderItemsService;

	protected Products $productsService;

	protected FrontProducts $frontProducts;

	public function __construct(int $orderId, Orders $orders, OrderItems $items, Products $products, FrontProducts $frontProducts)
	{
		$this->ordersService     = $orders;
		$this->orderItemsService = $items;
		$this->productsService   = $products;
		$this->frontProducts     = $frontProducts;
		$this->order             = $this->ordersService->get($orderId);
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ======================== Handler
	 */

	public function handleProductsLoad(): void
	{
		$output = [];
		$params = $this->getParameters();

		if (isset($params['site']))
			Sites::$currentIdentOverride = $params['site'];

		foreach ($this->productsService->getEr()->createQueryBuilder('p')
			         ->select('p.id, p.code1, p.ean, p.price, vr.rate as vatRate, pt.name')
			         ->innerJoin('p.vatRate', 'vr')
			         ->innerJoin('p.productTexts', 'pt', Join::WITH, 'pt.lang = :lang')
			         ->setParameter('lang', $this->translator->getLocale())
			         ->getQuery()->getArrayResult() as $row) {
			$output[$row['id']] = [
				'id'      => (string) $row['id'],
				'name'    => (string) $row['name'],
				'code1'   => (string) $row['code1'],
				'ean'     => (string) $row['ean'],
				'price'   => (float) $row['price'],
				'vatRate' => (int) $row['vatRate'],
			];
		}

		$this->getPresenter()->sendJson($output);
	}

	public function handleProductDetail(): void
	{
		$products = [];
		$params   = $this->getParameters();

		$arrProd = $this->productsService->getEr()->createQueryBuilder('p')
			->select('p.id, p.code1, p.ean, p.price, vr.rate as vatRate')
			->innerJoin('p.vatRate', 'vr')
			->where('p.id = :id')
			->setParameter('id', $params['id'])
			->getQuery()->getOneOrNullResult();

		$products[$arrProd['id']] = (new \EshopCatalog\FrontModule\Model\Dao\Product((int) $arrProd['id']))
			->setCode1((string) $arrProd['code1'])
			->setEan((string) $arrProd['ean'])
			->setPrice((float) $arrProd['price'])
			->setVatRate((int) $arrProd['vatRate']);
		$this->frontProducts->loadFresh($products, $params['userId'] ?? null);
		$product = $products[$arrProd['id']];

		$this->getPresenter()->sendJson([
			'id'      => (string) $product->getId(),
			'name'    => (string) $product->name,
			'code1'   => (string) $product->getCode1(),
			'ean'     => (string) $product->getEan(),
			'price'   => $product->priceInBaseCurrency,
			'vatRate' => $product->getVatRate(),
		]);
	}

	/*******************************************************************************************************************
	 * ======================== Components
	 */

	protected function createComponentForm()
	{
		$form = $this->createForm();
		$form->setAjax();
		$form->getElementPrototype()->addClass('orderItemAddForm');

		$linkParams = [];
		try {
			$linkParams['site'] = $this->order->site->getIdent();

			$firstSignIn = $this->order->getCustomer()->getUser()->getFirstSignIn();
			if ($firstSignIn !== null)
				$linkParams['userId'] = $this->order->getCustomer()->getUser()->getId();
		} catch (\Exception$e) {
		}

		$form->addHidden('productId')->setRequired('Vyberte prosím produkt');
		$form->addText('product', 'eshopOrders.default.productId')
			->setHtmlAttribute('data-autocomplete-name', 'orderItemAddForm-product')
			->setHtmlAttribute('data-autocomplete-url', $this->link('productsLoad!', $linkParams))
			->setHtmlAttribute('data-autocomplete-target', $form['productId']->getHtmlId())
			->setHtmlAttribute('data-autocomplete-keys', 'id,code1,ean,name')
			->setHtmlAttribute('data-detail-link', $this->link('productDetail!', $linkParams + ['id' => '__id__']))
			->setRequired();
		$form->addInteger('quantity', 'eshopOrders.default.quantity')->setRequired()->addRule(Form::MIN, 'Počet musí být alespoň %d', 1);
		$form->addText('code1', 'eshopOrders.itemsGrid.code1');
		$form->addText('price', 'eshopOrders.default.pricePiece')->setRequired()->addRule(Form::FLOAT, 'Cena musí být číslo');
		$form->addText('vatRate', 'eshopOrders.itemsGrid.vatRate')->setRequired()->addRule(Form::INTEGER, 'DPH musí být celé číslo');

		$form->addSaveCancelControl();
		$form->onSuccess[]  = [$this, 'formSuccess'];
		$form->onValidate[] = function(BaseForm $form, ArrayHash $values) {
			$this->redrawControl('formErrors');
		};

		$form->setDefaults([
			'quantity' => 1,
		]);

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
		try {
			//			bdump($values);
			//			die();
			$lang      = $this->translator->getLocale();
			$orderItem = new OrderItem($this->em->getReference(Product::class, $values->productId), $this->order);
			$orderItem->setCode1($values->code1);
			$orderItem->addOrderItemText($lang);
			$orderItem->getOrderItemText($lang)->setName($values->product);
			$orderItem->setQuantity($values->quantity);
			$orderItem->setPrice($values->price);
			$orderItem->setVatRate($values->vatRate);

			$this->em->persist($orderItem);

			$this->eventDispatcher->dispatch(new OrderItemEvent($orderItem), OrderItemAddForm::class . '::formSuccess');

			$this->em->flush();

			$this->getPresenter()->flashMessageSuccess('eshopOrders.itemForm.added');

			return true;
		} catch (\Exception $e) {
			$form->addError($e->getMessage());
			$this->redrawControl('form');
		}

		return false;
	}

	public function setOrder($orderId)
	{
		$this->order = $this->ordersService->get($orderId);
	}
}
