<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Order;

use Core\AdminModule\Model\Sites;
use Core\Model\Entities\Site;
use Core\Model\Event\Event;
use Core\Model\Event\GridFilterEvent;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\DataGrid\DataSource\DoctrineDataSource;
use Currency\DI\CurrencyExtension;
use Doctrine\Common\Collections\Criteria;
use Doctrine\ORM\Query;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\AdminModule\Model\Sellers;
use EshopOrders\AdminModule\Components\Expedition\IExpeditionAdvancedOptionsFormFactory;
use EshopOrders\AdminModule\Model\GroupsCustomers;
use EshopOrders\AdminModule\Model\Payments;
use EshopOrders\AdminModule\Model\Speditions;
use EshopOrders\AdminModule\Model\Statuses;
use EshopOrders\Model\Entities\GroupCustomers;
use EshopOrders\Model\Entities\Invoice;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderItem;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\Entities\Payment;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Event\OrderStatusEvent;
use EshopOrders\Model\Orders;
use Core\Model\Entities\QueryBuilder;
use EshopOrders\Model\OrderStatuses;
use EshopOrders\Model\Utils\Strings;
use Nette\Application\IPresenter;
use Nette\Http\Session;
use Nette\Http\SessionSection;
use Nette\Utils\DateTime;
use Nette\Utils\Html;
use Core\Model\Templating\Filters\Price as PriceFilter;
use Nette\Utils\Json;

class OrdersGrid extends BaseControl
{
	/** @var string @persistent */
	public $status = 'all';

	protected array $ids = [];

	/** @var array */
	protected $config;

	/** @var array */
	protected $exportCallbacks;

	protected Orders $ordersService;

	protected Speditions $speditionsService;

	protected Payments $paymentsService;

	protected Statuses $statusesService;

	protected OrderStatuses $orderStatusesService;

	protected Sites $sitesService;

	protected SessionSection $sessionSection;

	protected PriceFilter $priceFilter;

	protected Sellers $sellers;

	protected GroupsCustomers $groupsCustomers;

	protected IExpeditionAdvancedOptionsFormFactory $advancedOptionsFormFactory;

	public function __construct($data, Orders $orders, Speditions $speditions, Payments $payments, Statuses $statuses,
	                            Sites $sites, PriceFilter $priceFilter, Session $session, Sellers $sellers,
	                            OrderStatuses $orderStatusesService, GroupsCustomers $groupsCustomers, IExpeditionAdvancedOptionsFormFactory $advancedOptionsFormFactory)
	{
		$this->ordersService        = $orders;
		$this->config               = $data;
		$this->speditionsService    = $speditions;
		$this->paymentsService      = $payments;
		$this->statusesService      = $statuses;
		$this->sitesService         = $sites;
		$this->priceFilter          = $priceFilter;
		$this->sessionSection       = $session->getSection('eshopOrdersOrdersGrid');
		$this->status               = EshopOrdersConfig::load('ordersGrid.defaultStatus');
		$this->sellers              = $sellers;
		$this->orderStatusesService = $orderStatusesService;
		$this->groupsCustomers       = $groupsCustomers;
		$this->advancedOptionsFormFactory = $advancedOptionsFormFactory;

		if (EshopOrdersConfig::load('ordersGrid.useTabs'))
			$this->monitor(IPresenter::class, function() {
				$this->status = $this->sessionSection->status ?: 'created';
			});
	}

	public function render(): void
	{
		if (EshopOrdersConfig::load('ordersGrid.useTabs')) {
			$statuses                     = $this->statusesService->getForSelectOption();
			$this->template->statuses     = $statuses;
			$this->template->activeStatus = $this->status;
		}

		$this->template->render($this->getTemplateFile());
	}

	public function handleChangeStatus(string $s): void
	{
		$this->status                 = $s;
		$this->sessionSection->status = $s;
		$this->redrawControl('nav');
		$this['grid']->reload();
	}

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();
		$grid->setStrictSessionFilterValues(false)
			->setItemsPerPageList([20, 50, 100, 200], EshopOrdersConfig::load('ordersGrid.allowShowAllItems', false))
			->setDefaultPerPage(50);
		$grid->detailEyePosition = (string) EshopOrdersConfig::load('ordersGrid.detailEyePosition', 'right');
		foreach ($this->em->getRepository(Payment::class)->createQueryBuilder('p')
			         ->getQuery()->getResult() as $row) {
			$payments[$row->getId()] = $row;
		}

		$qb = $this->ordersService->getEr()->createQueryBuilder('o')
			->addSelect('ai, ad, payment, spedition, discounts, site, currency, customer, cGroup')
			->leftJoin('o.addressInvoice', 'ai')
			->leftJoin('o.addressDelivery', 'ad')
			->innerJoin('o.payment', 'payment')
			->innerJoin('o.spedition', 'spedition')
			->leftJoin('o.orderDiscounts', 'discounts')
			->leftJoin('o.currency', 'currency')
			->innerJoin('o.site', 'site')
			->leftJoin('o.customer', 'customer')
			->leftJoin('customer.groupCustomers', 'cGroup')
			->orderBy('o.id', 'DESC')
			->groupBy('o.id');

		$join = $this->em->getRepository(OrderStatus::class)->createQueryBuilder('os2')
			->select('GROUP_CONCAT(IDENTITY(os2.status))')
			->andWhere('os2.order = o.id')
			->andWhere('os2.deleted IS NULL OR os2.deleted = 0')
			->groupBy('os2.order')
			->orderBy('os2.order', 'DESC')
			->getQuery();

		if (EshopOrdersConfig::load('ordersGrid.useTabs')) {
			if (in_array($this->status, array_keys($this->statusesService->getForSelectOption())))
				$qb->addSelect("(" . $join->getDQL() . ") as hidden statusFilter")
					->andHaving('statusFilter LIKE :status2')
					->setParameter('status2', "%" . $this->status);
		}

		/** @var OrderStatus[][] $orderStatuses */
		$orderStatuses = [];
		/** @var Site $sites */
		$sites = [];
		/** @var OrderItem[][] $orderItems */
		$orderItems = [];
		/** @var Invoice $invoices */
		$invoices = [];
		/** @var Payment $payments */
		$payments = [];

		foreach ($this->sitesService->getAll() as $row)
			$sites[$row->getIdent()] = $row->getIdent();

		$grid->setDataSource(new DoctrineDataSource($qb, 'o.id'));

		$grid->getDataSource()->onDataLoaded[] = function($items) use (&$orderStatuses, &$orderItems, &$invoices, &$payments, $grid) {
			$ids            = [];
			$map            = [];
			$cardPaymentIds = [];
			$invoiceIds     = [];

			/** @var Order[] $items */
			foreach ($items as $k => $v) {
				$map[$v->getId()] = $k;
				$ids[]            = $v->getId();
				if ($v->getInvoice())
					$invoiceIds[$v->getInvoice()->getId()] = $v->getId();
			}
			$this->ids = $ids;

			foreach ($this->em->getRepository(OrderStatus::class)->createQueryBuilder('os')
				         ->addSelect('s')
				         ->innerJoin('os.status', 's')
				         ->where('os.order IN (:orders)')->setParameter('orders', $ids)
				         ->orderBy('os.created', 'DESC')
				         ->getQuery()->getResult() as $row) {
				/** @var OrderStatus $row */
				if ($row->isDeleted())
					continue;

				$orderStatuses[$row->getOrder()->getId()][] = $row;
			}

			foreach ($this->em->getRepository(OrderItem::class)->createQueryBuilder('oi')
				         ->addSelect('s, ch')
				         ->where('oi.order IN (:orders)')->setParameter('orders', $ids)
				         ->leftJoin('oi.sales', 's')
				         ->leftJoin('oi.children', 'ch')
				         ->getQuery()->getResult() as $row)
				$orderItems[$row->getOrder()->getId()][] = $row;

			$grid->template->orderItems = $orderItems;

			if (EshopOrdersConfig::load('invoice.enable')) {
				foreach ($this->em->getRepository(Invoice::class)->createQueryBuilder('i')
					         ->where('i.id IN (:ids)')
					         ->setParameter('ids', array_keys($invoiceIds))
					         ->getQuery()
					         ->setHint(Query::HINT_FORCE_PARTIAL_LOAD, true)
					         ->getResult() as $row) {
					/** @var Invoice $row */
					$invoices[$invoiceIds[$row->getId()]] = $row;
				}
			}

			$grid->template->loadItemsMoreData = function($items) {
				$event                     = new Event();
				$event->data['orderItems'] = &$items;
				$this->eventDispatcher->dispatch($event, 'eshopOrders.orderItems.loadMoreData');
			};
		};

		$customerLinkFn = fn($col) => function(Order $order) use ($col) {
			$ai = $order->getAddressInvoice();
			if ($order->getCustomer() && $order->getAddressInvoice()) {

				return Html::el('a', ['href' => $this->presenter->link('Customers:editCustomer', $order->getCustomer()->getId())])
					->setText($col === 'firstName' ? $ai->getFirstName() : $ai->getLastName());
			} else if ($order->getAddressInvoice()) {
				return $col === 'firstName' ? $ai->getFirstName() : $ai->getLastName();
			}
		};

		//Columns
		$grid->addColumnText('id', 'eshopOrders.default.orderNumberShort')->setAlign('right')
			->setRenderer(function(Order $order) {
				$html = Html::el('div')
					->addHtml(Html::el('a', ['href' => $this->getPresenter()->link('Default:editOrder', $order->getId())])
						->setText($order->getId()));

				$event = new Event(['html' => $html, 'order' => $order, 'allIds' => $this->ids]);
				$this->eventDispatcher->dispatch($event, OrdersGrid::class . '::columnIdRender');

				return $html;
			});
		$grid->addColumnText('firstName', 'default.name', 'addressInvoice.firstName')
			->setRenderer($customerLinkFn('firstName'));
		$grid->addColumnText('lastName', 'eshopOrders.default.lastName', 'addressInvoice.lastName')
			->setRenderer($customerLinkFn('lastName'));
		$grid->addColumnText('email', 'eshopOrders.default.email')->setRenderer(function(Order $row) {
			return Html::el('a', ['href' => 'mailto:' . $row->getAddressInvoice()->getEmail(), 'target' => '_blank'])
				->setText($row->getAddressInvoice()->getEmail());
		});
		$grid->addColumnPhone('phone', 'eshopOrders.default.phone', 'addressInvoice.phone', 'addressInvoice.country.id');

		if (EshopOrdersConfig::load('ordersGrid.showCustomerGroup', true)) {
			$grid->addColumnText('cGroup', 'eshopOrders.default.customerGroup')->setRenderer(function(Order $row) {
				return $row->getCustomer() && $row->getCustomer()->getGroupCustomers() ? $row->getCustomer()->getGroupCustomers()->short : '';
			})->setAlign('center');

			$grid->getColumn('cGroup')->getElementPrototype('td')->class[] = 'w1nw';
		}

		$grid->addColumnText('spedition', 'eshopOrders.default.spedition', 'spedition.name')->setRenderer(function(Order $order) {
			$html = Html::el()->setText($order->getSpedition()->getName());

			$event = new Event(['html' => $html, 'order' => $order, 'allIds' => $this->ids]);
			$this->eventDispatcher->dispatch($event, OrdersGrid::class . '::columnSpeditionRender');

			return $html;
		});
		$grid->addColumnText('payment', 'eshopOrders.default.payment', 'payment.name');
		$grid->addColumnDateTime('created', 'eshopOrders.defaultGrid.createdTime', 'createdTime')
			->setRenderer(function(Order $row) use (&$orderStatuses, &$invoices) {
				if ($row->isCorrectiveTaxDocument && isset($invoices[$row->getId()])) {
					return $invoices[$row->getId()]->createdAt->format('d. m. Y H:i');
				}

				$statuses = $orderStatuses[$row->getId()];

				return $statuses ? end($statuses)->getCreated()->format('d. m. Y H:i') : '';
			})->getElementPrototype('th')->addClass('w1nw');

		// Stavy objednavky
		$statusesShowSeparated  = EshopOrdersConfig::load('ordersGrid.showStatusesSeparated', false);
		$statusesUseShortTitles = EshopOrdersConfig::load('ordersGrid.showShortStatusNames', false);
		$statusesTitlePrefix    = 'eshopOrders.ordersGrid.' . ($statusesUseShortTitles ? 'statusesShort.' : 'statusesLong.');
		if (is_array($statusesShowSeparated)) {
			foreach ($statusesShowSeparated as $key) {
				$grid->addColumnText('status' . ucfirst($key), $statusesTitlePrefix . $key)
					->setRenderer(function(Order $row) use ($key, &$orderStatuses, &$payments) {
						$paymentIdent = $row->getPaymentIdent();

						$status     = false;
						$statuses   = $orderStatuses[$row->getId()];
						$lastStatus = $statuses[0]->getStatus()->getId();

						if ($lastStatus === OrderStatus::STATUS_CANCELED) {
							$status = 'bg-success';
						} else {
							if ($key === OrderStatus::STATUS_IS_PAID) {
								if ($row->isPaid) {
									$status = 'bg-success';
								} else if (in_array((string) $paymentIdent, ['card', 'transfer', 'pickup',
									'store', 'storeCash', 'storeCard'])) {
									$status = 'bg-danger';
								}
							} else if ($key === OrderStatus::STATUS_SPEDITION && $lastStatus === OrderStatus::STATUS_PROCESSING) {
								$status = 'bg-warning';
							} else {
								foreach ($statuses as $orderStatus) {
									if ($orderStatus->getStatus()->getId() === $key) {
										$status = 'bg-success';
										break;
									}
								}
							}
						}

						if ($status) {
							return Html::el('div', [
								'class' => 'status-circle ' . $status,
							]);
						}

						return '';
					})
					->setAlign('center')
					->setFilterSelect([
						''  => '',
						'0' => $this->t('default.no'),
						'1' => $this->t('default.yes'),
					])->setCondition(function(QueryBuilder $qb, $value) use ($key) {
						if ($key === 'isPaid') {
							$qb->andWhere('o.isPaid = :osIsPaid')
								->setParameter('osIsPaid', $value);
						} else {
							$queryKey = 'os' . ucfirst($key);

							if ($value) {
								$qb->innerJoin('o.orderStatuses', $queryKey, Join::WITH, "$queryKey.status = :$queryKey")
									->setParameter($queryKey, $key);
							} else {
								if (!in_array('osConcat', $qb->getAllAliases()))
									$qb->leftJoin('o.orderStatuses', 'osConcat')->addSelect('GROUP_CONCAT(IDENTITY(osConcat.status)) as hidden statusesString');
								$qb->andHaving("statusesString NOT LIKE :$queryKey")
									->setParameter($queryKey, "%$key%");
							}
						}
					});
			}
		}

		if (EshopOrdersConfig::load('ordersGrid.useTabs', false) === false
			&& EshopOrdersConfig::load('ordersGrid.showCurrentStatus', true) === true) {
			$grid->addColumnText('status', 'eshopOrders.default.status')->setRenderer(function(Order $row) use (&$orderStatuses) {
				$statuses = $orderStatuses[$row->getId()];

				return $statuses ? $statuses[0]->getStatus()->getName() : '';
			});
		}

		if (EshopOrdersConfig::load('ordersGrid.showCardHistory')
			&& (!is_array($statusesShowSeparated) || !in_array('isPaid', $statusesShowSeparated))) {
			$showPaidFieldForAll = EshopOrdersConfig::load('showPaidFieldForAllOrders', false);
			$grid->addColumnText('isPaid', 'eshopOrders.default.' . ($showPaidFieldForAll ? 'isPaid' : 'isPaidWithCard'))
				->setRenderer(function(Order $row) use ($showPaidFieldForAll) {
					if (!($showPaidFieldForAll || $row->getPayment()->getPayment()->getIdent() === 'card'))
						return '';

					$isPaid = $row->isPaid;

					return Html::el('div', [
						'class' => 'btn btn-xs ' . ($isPaid ? 'btn-success' : 'btn-danger'),
					])->setText($this->t('default.' . ($isPaid ? 'yes' : 'no')));
				})
				->setAlign('center')
				->setFilterSelect([
					''  => '',
					'0' => $this->t('default.no'),
					'1' => $this->t('default.yes'),
				]);
		}

		$grid->addColumnNumber('price', 'eshopOrders.default.price')->setRenderer(function(Order $row) use (&$orderItems) {
			if ($orderItems[$row->getId()])
				$row->setOrderItems($orderItems[$row->getId()]);

			if ($row->currency) {
				$price2    = $row->getPrice(true);
				$code2     = $row->currency->code;
				$decimals2 = $row->currency->decimals;
			}

			return $this->priceFilter
				->renderTwoPrices($row->getPrice(), $row->getDefaultCurrency(), $price2 ?? null, $code2 ?? null, $decimals2 ?? null);
		})->getElementPrototype('th')->addClass('w1nw');

		if (class_exists(CurrencyExtension::class)) {
			$grid->addColumnText('currency', 'eshopOrders.default.currency', 'currencyCode');
		}

		if (count($sites) > 1) {
			$grid->addColumnText('site', 'eshopOrders.default.eshop', 'site.ident');

			$tmp     = [];
			$sellers = [];
			foreach ($this->sellers->getSitesInUse() as $row)
				$sellers[$row['sellerName']][] = $row['site'];

			foreach (EshopOrdersConfig::load('ordersGrid.extraSellersFilter', []) as $k => $v)
				$sellers[$k] = $v;

			foreach ($sellers as $k => $v)
				$tmp['sellers_' . implode('|', $v)] = $k;

			$grid->getColumn('site')->setFilterSelect([
				null                                       => '',
				$this->t('eshopOrders.ordersGrid.eshops')  => $sites,
				$this->t('eshopOrders.ordersGrid.sellers') => $tmp,
			])->setCondition(function(QueryBuilder $qb, $value) {
				if (Strings::startsWith($value, 'sellers_')) {
					$value = explode('|', substr($value, 8));
				} else {
					$value = [$value];
				}
				$qb->andWhere('site.ident ' . (count($value) > 1 ? 'IN (:sites)' : '= :sites'))
					->setParameter('sites', count($value) > 1 ? $value : $value[0]);
			});
		}

		if (EshopOrdersConfig::load('invoice.enable') && EshopOrdersConfig::load('ordersGrid.showInvoiceDueDate')) {
			$grid->addColumnText('invoiceDueDate', 'eshopOrders.ordersGrid.invoiceDueDate')
				 ->setRenderer(function (Order $o) {
					 if ($o->getInvoice() === null) {
					 	return '';
					 }

					 $now = (new DateTime)->setTime(0, 0);
					 $dueDate = (clone $o->getInvoice()->dueDate)->setTime(0, 0);

					 return Html::el('span', ['class' => $now->getTimestamp() > $dueDate->getTimestamp() && !$o->isPaid ? 'text-danger' : ''])
						 ->setText($dueDate->format('d.m.Y'));
				 });
		};

		// Filter
		$grid->addFilterText('products', 'eshopOrders.ordersGrid.searchInProducts')->setCondition(function(QueryBuilder $qb, $value) {
			$qb->innerJoin('o.orderItems', 'ordItm')
				->innerJoin('ordItm.orderItemTexts', 'ordItmTxt')
				->andWhere('ordItm.product = :products OR ordItm.code1 LIKE :productsL OR ordItmTxt.name LIKE :productsL')
				->setParameter('products', $value)
				->setParameter('productsL', "%{$value}%");
		});
		$grid->addFilterText('packageNumber', 'eshopOrders.ordersGrid.searchPackageNumber')->setCondition(function(QueryBuilder $qb, $value) {
			$criteria = Criteria::create();
			$this->eventDispatcher->dispatch(new GridFilterEvent($qb, $criteria, $value), OrdersGrid::class . '::filterPackageNumber');
			$qb->addCriteria($criteria);
		});
		$grid->addFilterText('id', '')->setCondition(function(QueryBuilder $qb, $value) {
			$criteria = Criteria::create();
			$criteria->orWhere(Criteria::expr()->eq('o.id', $value));

			$this->eventDispatcher->dispatch(new GridFilterEvent($qb, $criteria, $value), OrdersGrid::class . '::filterId');
			$qb->addCriteria($criteria);
		});
		$grid->addFilterText('firstName', '', ['ad.firstName', 'ai.firstName']);
		$grid->addFilterText('lastName', '', ['ad.lastName', 'ai.lastName']);
		$grid->addFilterText('email', '', ['ad.email', 'ai.email']);
		$grid->addFilterText('phone', '', ['ad.phone', 'ai.phone']);
		$grid->addFilterSelect('spedition', '', ['' => ''] + $this->speditionsService->getForSelectOption(), 'spedition.spedition');
		$grid->addFilterSelect('payment', '', ['' => ''] + $this->paymentsService->getForSelectOption(), 'payment.payment');
		$grid->addFilterDateRange('created', '')->setCondition(function(QueryBuilder $qb, $values) {
			if ($values->from || $values->to) {
				if (!in_array('oStatuses', $qb->getAllAliases()))
					$qb->innerJoin('o.orderStatuses', 'oStatuses');
				$qb->andWhere('oStatuses.status = :statusName2')->setParameter('statusName2', 'created');
			}

			if ($values->from)
				$qb->andWhere('oStatuses.created >= :statusFrom')
					->setParameter('statusFrom', DateTime::createFromFormat('j. n. Y', $values->from)->setTime(0, 0, 0));
			if ($values->to)
				$qb->andWhere('oStatuses.created <= :statusTo')
					->setParameter('statusTo', DateTime::createFromFormat('j. n. Y', $values->to)->setTime(23, 59, 59));
		});
		$grid->addFilterRange('price', '')->setCondition(function(QueryBuilder $qb, $values) {
			$qb->innerJoin('o.orderItems', 'items')
				->addSelect('SUM(items.price) + payment.price + spedition.price + IFNULL(SUM(discounts.price), 0) as hidden hPrice');

			if ($values->from)
				$qb->andHaving('hPrice >= :priceFrom')->setParameter('priceFrom', (float) $values->from);
			if ($values->to)
				$qb->andHaving('hPrice <= :priceTo')->setParameter('priceTo', (float) $values->to);
		});

		if (isset($grid->getColumns()['status'])) {
			$grid->addFilterSelect('status', '', ['' => ''] + $this->statusesService->getForSelectOption())->setCondition(function(QueryBuilder $qb, $value) {
				$join = $this->em->getRepository(OrderStatus::class)->createQueryBuilder('os2')
					->select('GROUP_CONCAT(IDENTITY(os2.status))')
					->andWhere('os2.order = o.id')
					->andWhere('os2.deleted IS NULL OR os2.deleted = 0')
					->groupBy('os2.order')
					->orderBy('os2.order', 'DESC')
					->getQuery();

				$qb->addSelect("(" . $join->getDQL() . ") as hidden statusFilter")
					->andHaving('statusFilter LIKE :status2')
					->setParameter('status2', "%$value");
			});
			$grid->getColumn('status')->getElementPrototype('th')->class[] = 'w1nw';
		}

		if (isset($grid->getColumns()['cGroup'])) {
			$grid->addFilterSelect('cGroup', '', ['' => ''] + ['withoutGroup' => $this->t('eshopOrders.default.withoutGroup')] + $this->groupsCustomers->getOptionsForSelect())->setCondition(function(QueryBuilder $qb, $value) {
				if ($value === 'withoutGroup') {
					$qb->andWhere('cGroup IS NULL');
				} else {
					$qb->andWhere('cGroup = :group')
						->setParameter('group', $value);
				}
			});
			$grid->getColumn('cGroup')->getElementPrototype('th')->class[] = 'w1nw';
		}

		$grid->setHeadFilters(['products', 'packageNumber']);

		// ItemDetail
		$grid->setItemsDetail($this->getTemplateFile('OrdersGridDetail', 'Order'));

		// Actions
		$grid->addAction('edit', '', 'Default:editOrder')->setIcon('edit')->setBsType('primary');
		if ($this->config['allowOrderDelete']) {
			$grid->addAction('delete', '', 'delete!')->setConfirm('eshopOrders.orderForm.reallyDelete')->setIcon('times')->setBsType('danger');
		}

		$grid->addGroupSelectCustomAction('eshopOrders.defaultGrid.changeStatus', $this->statusesService->getForSelectOption() + [
				OrderStatus::STATUS_IS_PAID => 'eshopOrders.ordersGrid.statusesLong.isPaid',
			])->onSelect[] = [$this, 'gridChangeStatus'];
		if ($this->exportCallbacks) {
			foreach ($this->exportCallbacks as $k => $v) {
				if (EshopOrdersConfig::load('expeditionOrdersGrid.enableAdvancedOptions', false) && Strings::contains($k, 'ceskaposta') && !Strings::endsWith($k, 'Labels')) {
					$grid->addGroupAction($v['title'])->onSelect[] = $v['callback'];
				} else if (!Strings::endsWith($k, 'Labels')) {
					$grid->addGroupTextAction($v['title'])
						 ->setAttribute('placeholder', $this->t('eshopOrders.expeditionOrdersGrid.form.quantity'))
						 ->onSelect[] = $v['callback'];
				} else {
					$grid->addGroupAction($v['title'])->onSelect[] = $v['callback'];
				}
			}
		}

		$grid->addAction('pdf', '')->setRenderer(function(Order $order) {
			if ($order->getInvoice() === null) {
				return '';
			}

			return Html::el('a', ['href'  => $this->presenter->link('Orders:invoice', $order->getId()),
			                      'class' => 'btn btn-danger btn-xs', 'target' => '_blank'])
				->addHtml(Html::el('i', ['class' => 'fas fa-file-invoice']));
		});

		$grid->addGroupAction('eshopOrders.ordersGrid.printOrders')->onSelect[] = function(array $ids) {
			$presenter                          = $this->getPresenter();
			$presenter->payload->fileRequests[] = [
				'name' => 'print',
				'url'  => $presenter->link(':EshopOrders:Admin:Orders:print', ['ids' => $ids]),
			];

			$presenter->sendPayload();
		};

		if (EshopOrdersConfig::load('enableDeliveryListPrint')) {
			$grid->addGroupAction('eshopOrders.ordersGrid.printDeliveryList')->onSelect[] = function(array $ids) {
				$presenter                          = $this->getPresenter();
				$presenter->payload->fileRequests[] = [
					'name' => 'print',
					'url'  => $presenter->link(':EshopOrders:Admin:Orders:printDeliveryList', ['ids' => $ids]),
				];

				$presenter->sendPayload();
			};
		}

		// Columns prototype
		$grid->getColumn('id')->getElementPrototype('th')->class[]      = 'w1nw';
		$grid->getColumn('created')->getElementPrototype('td')->class[] = 'w1nw';
		$grid->getColumn('phone')->getElementPrototype('td')->class[]   = 'w1nw';

		return $grid;
	}

	protected function createComponentExpeditionAdvancedOptionsForm()
	{
		return $this->advancedOptionsFormFactory->create();
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleDelete($id)
	{
		$presenter = $this->getPresenter();
		if ($this->config['allowOrderDelete']) {
			if ($this->ordersService->remove($id))
				$presenter->flashMessageSuccess('default.removed');
			else
				$presenter->flashMessageDanger('default.removeFailed');
		} else {
			$presenter->flashMessageDanger('eshopOrders.orderForm.removeDenied');
		}

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	/*******************************************************************************************************************
	 * =================  Grid function
	 */

	public function gridChangeStatus(array $ids, string $value): void
	{
		if (!$value || empty($ids))
			return;

		$error     = true;
		$presenter = $this->getPresenter();

		foreach ($ids as $orderId) {
			$order = $this->ordersService->getReference($orderId);
			if (!$order) {
				continue;
			}
			$event            = new OrderStatusEvent($order, $value);
			$event->presenter = $presenter;
			$this->eventDispatcher->dispatch($event, OrdersGrid::class . '::changeStatus');
		}

		if ($value === OrderStatus::STATUS_IS_PAID) {
			if ($this->ordersService->setPaid($ids))
				$error = false;
		} else if ($this->statusesService->changeStatus($ids, $value))
			$error = false;

		if ($error)
			$presenter->flashMessageDanger('eshopOrders.defaultGrid.publishChangeFailed');
		else
			$presenter->flashMessageSuccess('eshopOrders.defaultGrid.publishChanged');

		if ($presenter->isAjax()) {
			$presenter->redrawControl('flashes');
			$this['grid']->reload();
		} else {
			$this->redirect('this');;
		}
	}

	/*******************************************************************************************************************
	 * =================  GET SET
	 */

	public function addExportCallback(string $key, string $title, callable $callback): self
	{
		$this->exportCallbacks[$key] = [
			'title'    => $title,
			'callback' => $callback,
		];

		return $this;
	}

	/*******************************************************************************************************************
	 * =================  OTHERS
	 */

	public function openExpeditionAdvancedOptionsFormModal(array $orderIds, string $speditionIdent): void
	{
		$this['expeditionAdvancedOptionsForm']->setDefaults($orderIds, $speditionIdent);
		$this->template->modal = 'expeditionAdvancedOptionsForm';
		$this->template->modalTitle = $this->t('eshopOrders.expeditionAdvancedOptionsForm.title');
		$this->redrawControl('modal');
	}

}
