<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Model\Statistics;

use DateTimeInterface;
use Doctrine\ORM\Query;
use Doctrine\ORM\QueryBuilder;
use EshopOrders\Model\Entities\Order;
use Core\Model\Entities\EntityManagerDecorator;
use EshopOrders\Model\Entities\OrderStatus;
use Nette\Utils\DateTime;

class Statistics
{
	protected EntityManagerDecorator $em;

	protected ?array $cTodayStatusIds = null;

	protected ?array $cTodayCreated = null;

	public function __construct(EntityManagerDecorator $em)
	{
		$this->em = $em;
		$this->em->getConfiguration()->setSQLLogger(null);
	}

	public function getTodayStatusIds(string $status): array
	{
		if (!isset($this->cTodayStatusIds[$status])) {
			$this->cTodayStatusIds[$status] = [];

			foreach ($this->em->getRepository(OrderStatus::class)->createQueryBuilder('os')
				         ->select('IDENTITY(os.order) as order, os.id')
				         ->where('os.status = :status')
				         ->andWhere('os.created >= :today')
				         ->setParameters([
					         'status' => $status,
					         'today'  => (new DateTime())->setTime(0, 0, 0),
				         ])->getQuery()->getArrayResult() as $row)
				$this->cTodayStatusIds[$status][$row['order']] = $row;
		}

		return $this->cTodayStatusIds[$status];
	}

	public function getBaseQueryBuilder(): QueryBuilder
	{
		return $this->em->getRepository(Order::class)->createQueryBuilder('o', 'o.id')
			->addSelect('p, s, oi, oiSales, os, curr, od, pp, ss, site, inv, corDoc')
			->join('o.payment', 'p')
			->leftJoin('p.payment', 'pp')
			->join('o.spedition', 's')
			->leftJoin('s.spedition', 'ss')
			->innerJoin('o.site', 'site')
			->join('o.orderItems', 'oi')
			->leftJoin('oi.sales', 'oiSales')
			->innerJoin('o.orderStatuses', 'os')
			->leftJoin('o.currency', 'curr')
			->leftJoin('o.orderDiscounts', 'od')
			->leftJoin('o.addressInvoice', 'inv')
			->leftJoin('o.orderForCorrectiveTaxDocument', 'corDoc');
	}

	/**
	 * @return Order[]
	 */
	public function getTodayCreated(): array
	{
		if ($this->cTodayCreated === null) {
			$this->cTodayCreated = [];

			foreach ($this->getBaseQueryBuilder()
				         ->andWhere('os.created >= :today AND os.status = :status')
				         ->setParameters([
					         'status' => OrderStatus::STATUS_CREATED,
					         'today'  => (new DateTime())->setTime(0, 0, 0),
				         ])
				         ->groupBy('o.id')
				         ->getQuery()->getResult() as $order) {
				/** @var Order $order */
				$this->cTodayCreated[$order->getId()] = $order;
			}
		}

		return $this->cTodayCreated;
	}

	public function getTodayOrders(): array
	{
		$result = [
			'count' => 0,
			'price' => 0,
		];

		$canceled = $this->getTodayStatusIds(OrderStatus::STATUS_CANCELED);

		foreach ($this->getTodayCreated() as $order) {
			if (isset($canceled[$order->getId()]))
				continue;

			$result['count']++;
			$result['price'] += $order->getPriceItemsDiscount();
		}

		return $result;
	}

	public function getMonthly(?DateTimeInterface $from = null, ?DateTimeInterface $to = null)
	{
		$from = $from ?: (new DateTime())->modify('-1 year');
		$to   = $to ?: (new DateTime());

		$from->modify('first day of this month')->setTime(0, 0, 0);
		$to->modify('last day of this month')->setTime(23, 59, 59);

		$result = [];

		$allOrders = $this->getBaseQueryBuilder()
			->andWhere('os.status = :status')
			->andWhere('os.created >= :from')->andWhere('os.created <= :to')
			->setParameters([
				'from'   => $from,
				'to'     => $to,
				'status' => OrderStatus::STATUS_CREATED,
			])->getQuery()
			->setHint(Query::HINT_FORCE_PARTIAL_LOAD, 1)
			->getResult();

		$statuses = [];
		foreach ($this->em->getRepository(OrderStatus::class)->createQueryBuilder('os')
			         ->select('IDENTITY(os.order) as order, IDENTITY(os.status) as status, os.created')
			         ->where('os.order IN (:ids)')
			         ->andWhere('os.deleted IS NULL')
			         ->setParameter('ids', array_keys($allOrders))
			         ->getQuery()->getArrayResult() as $row) {
			$statuses[$row['order']][] = $row;
		}

		foreach ($allOrders as $order) {
			/** @var Order $order */
			$created = null;

			if (!isset($statuses[$order->getId()]))
				continue;

			foreach ($statuses[$order->getId()] as $s) {
				if ($s['status'] == OrderStatus::STATUS_CREATED)
					$created = $s['created'];
				else if ($s['status'] == OrderStatus::STATUS_CANCELED)
					continue 2;
			}

			if ($created) {
				$key = $created->format('Y-m');
				if (!isset($result[$key]))
					$result[$key] = $this->monthlyCreateRow($created);

				$result[$key]['orders']++;
				$result[$key]['ordersPrice'] += $order->getPriceItemsDiscount();
			}
		}

		krsort($result, SORT_STRING);

		return $result;
	}

	protected function monthlyCreateRow(\DateTime $dateTime)
	{
		return [
			'date'          => $dateTime,
			'orders'        => 0,
			'ordersPrice'   => 0,
			'canceled'      => 0,
			'canceledPrice' => 0,
		];
	}
}
