<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Components\Cart;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\Model\Config;
use EshopOrders\FrontModule\Model\Carts;
use EshopOrders\FrontModule\Model\Dao\CartItem;
use EshopOrders\FrontModule\Model\Event\UpdatedCartItemEvent;
use Nette\Utils\ArrayHash;

class CartDetailItem extends BaseControl
{
	/** @var Carts */
	protected $cartsService;

	/** @var int */
	protected $itemId;

	public function __construct(Carts $carts)
	{
		$this->cartsService = $carts;
	}

	public function render(): void
	{
		$this->template->item = $this->getItem();
		$this->template->render($this->getTemplateFile(null, 'CartDetailItem'));
	}

	protected function createComponentForm(): ?BaseForm
	{
		$form = $this->createForm();
		$item = $this->getItem();

		$form->addText('quantity', null)
			->addRule($form::FILLED)
			->setHtmlAttribute('data-add-to-cart-quantity-input', 'instant')
			->setHtmlAttribute('data-item-id', $item->getId())
			->setDefaultValue($item->quantity);
		$form->addHidden('itemId', (string) $item->getId());

		$product = $item->product;
		if ($product && Config::load('pseudoWarehouse') && $product->unlimitedQuantity === 0 && $product->getQuantity() > 0) {
			$form['quantity']->addRule(BaseForm::MAX, 'eshopCatalogFront.cart.maximumQuantity', $product->getQuantity());
			$form['quantity']->setHtmlAttribute('data-max', $product->getQuantity());
		}

		if ($item->getData('note')) {
			$form->addTextArea('note', $item->getData('noteTitle'))
				->setDefaultValue($item->getData('note'));
		}

		$childsContainer = $form->addContainer('childs');
		foreach ($item->getChilds() as $child) {
			$childContainer = $childsContainer->addContainer('child' . $child->getId());

			if ($child->getData('note'))
				$childContainer->addTextArea('note', $child->getData('noteTitle'))
					->setDefaultValue($child->getData('note'));
		}

		$form->onSuccess[] = [$this, 'formOnSuccess'];

		return $form;
	}

	public function formOnSuccess($form, ArrayHash $values): bool
	{
		try {
			$cartItems = $this->cartsService->getCurrentCart()->getCartItems();
			$event     = new UpdatedCartItemEvent(
				(int) $values->itemId,
				(int) $values->quantity,
				(int) $cartItems[$values->itemId]->quantity,
			);

			if ($values->note)
				$event->moreData['note'] = $values->note;

			$this->eventDispatcher->dispatch($event, 'eshopOrders.cartUpdateItem');

			foreach ($values->childs ?? [] as $childId => $childVals) {
				if (empty((array) $childVals))
					continue;

				$childId = substr($childId, 5);
				$child   = $cartItems[$values->itemId]->getChilds()[$childId];

				$event = new UpdatedCartItemEvent(
					(int) $child->getId(),
					(int) $child->quantity,
					(int) $child->quantity,
				);

				if ($childVals->note)
					$event->moreData['note'] = $childVals->note;

				bdump($event);
				$this->eventDispatcher->dispatch($event, 'eshopOrders.cartUpdateItem');
			}
		} catch (\Exception $e) {
			$this->getPresenter()->flashMessageWarning('eshopOrders.cart.countUpdateErr');
			$this->getPresenter()->redrawControl('flashes');

			return false;
		}

		return true;
	}

	protected function getItem(): ?CartItem
	{
		return $this->cartsService->getCartItem($this->itemId);
	}

	public function setItem(int $itemId): self
	{
		$this->itemId = $itemId;

		return $this;
	}
}

