<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Model;

use Core\Model\Event\EventDispatcher;
use Core\Model\Helpers\BaseFrontEntityService;
use Core\Model\Sites;
use EshopOrders\FrontModule\Model\Event\PaymentsSpeditionsEvent;
use EshopOrders\Model\Entities\PaymentSpedition;
use EshopOrders\Model\Entities\Spedition;
use EshopOrders\FrontModule\Model\Dao;
use Doctrine\ORM\Query\Expr\Join;
use EshopOrders\Model\Entities\SpeditionPriceLevel;
use EshopOrders\Model\EshopOrdersConfig;
use Users\Model\Security\User;

/**
 * Class Speditions
 * @package EshopOrders\FrontModule\Model
 */
class Speditions extends BaseFrontEntityService
{
	/** @var string */
	protected $entityClass = Spedition::class;

	/** @var EventDispatcher */
	public EventDispatcher $eventDispatcher;

	/** @var Sites */
	public Sites $sitesService;

	/** @var User */
	protected User $user;

	/** @var Customers */
	protected Customers $customers;

	/** @var Dao\Spedition[] */
	protected $cPublished;

	/** @var SpeditionPriceLevel[]|null */
	protected ?array $cGroupPriceLevels = null;

	public function __construct(EventDispatcher $eventDispatcher, Sites $sitesService, User $user, Customers $customers)
	{
		$this->eventDispatcher = $eventDispatcher;
		$this->sitesService    = $sitesService;
		$this->user            = $user;
		$this->customers       = $customers;
	}

	/**
	 * @param int $id
	 *
	 * @return Dao\Spedition
	 */
	public function get($id)
	{
		return $this->getAllPublished()[$id];
	}

	/**
	 * @return Dao\Spedition[]
	 */
	public function getAllPublished(): array
	{
		if ($this->cPublished === null) {
			$this->cPublished = [];
			$qb               = $this->getEr()->createQueryBuilder('s', 's.id')
				->andWhere('s.isPublished = :isPublished')
				->innerJoin(PaymentSpedition::class, 'ps', Join::WITH, 'ps.spedition = s.id')
				->setParameters([
					'isPublished' => 1,
				])
				->orderBy('s.position')
				->groupBy('s.id');

			if (count($this->sitesService->getSites()) > 1) {
				$qb->innerJoin('ps.sites', 'sites', Join::WITH, 'sites.ident = :site')
					->setParameter('site', $this->sitesService->getCurrentSite()->getIdent());
			}

			$customer = $this->user->isLoggedIn() ? $this->customers->getByUserId($this->user->getId()) : null;
			$customer = $this->user->isLoggedIn() ? $this->customers->getByUserId($this->user->getId()) : null;
			if ($customer) {
				if (EshopOrdersConfig::load('enableDisablingPaymentSpeditionForCustomer', false) && $customer->disabledSpeditions->getKeys()) {
					$qb->andWhere('s.id NOT IN (:notIds)')
						->setParameter('notIds', $customer->disabledSpeditions->getKeys());
				}

				if (EshopOrdersConfig::load('enablePriceLevels', false)) {
					$group = $customer->getGroupCustomers() ?: null;

					if ($group) {
						$qb->leftJoin('s.priceLevels', 'sPriceLevels', Join::WITH, 'sPriceLevels.group = :group')
							->setParameter('group', $group->getId())
							->addSelect('sPriceLevels');
					}
				}
			}

			$this->cPublished = $this->fillDao($qb->getQuery()->getArrayResult());
		}

		return $this->cPublished;
	}

	/**
	 * @param Dao\Spedition[] $speditions
	 * @param Dao\Cart        $cart
	 *
	 * @return Dao\Spedition[]
	 */
	public function filterByCart(array $speditions, Dao\Cart $cart): array
	{
		$cartValue        = $cart->getCartItemsPrice();
		$disabledPickUp   = $cart->hasDisabledPickUpSpedition();
		$oversizedProduct = $cart->hasOversizedProduct();

		foreach ($speditions as $k => $spedition) {
			if ($spedition->getAvailableFrom() > $cartValue || $spedition->getAvailableTo() < $cartValue)
				unset($speditions[$k]);

			if ($disabledPickUp && $spedition->isPickup === true
				|| $oversizedProduct && $spedition->allowOversized === false)
				unset($speditions[$k]);
		}

		return $speditions;
	}

	/**
	 * @param int $id
	 *
	 * @return \Doctrine\ORM\Proxy\Proxy|Spedition|null
	 */
	public function getReference(int $id) { return $this->getEr()->getReference($id); }

	/**
	 * @param array[] $speditionsRaw
	 *
	 * @return Dao\Spedition[]
	 */
	public function fillDao($speditionsRaw)
	{
		$speditions = [];
		foreach ($speditionsRaw as $spedRaw) {
			$spedition  = new Dao\Spedition();
			$priceLevel = EshopOrdersConfig::load('enablePriceLevels', false) && $spedRaw['priceLevels']
				? array_values($spedRaw['priceLevels'])[0] ?? null
				: null;

			$spedition
				->setId((int) $spedRaw['id'])
				->setIdent($spedRaw['ident'])
				->setName($spedRaw['name'])
				->setText($spedRaw['text'])
				->setPublished((int) $spedRaw['isPublished'])
				->setPosition((int) $spedRaw['position'])
				->setAvailableFrom($priceLevel ? $priceLevel['availableFrom'] : $spedRaw['availableFrom'])
				->setAvailableTo($priceLevel ? $priceLevel['availableTo'] : $spedRaw['availableTo'])
				->setImage($spedRaw['image']);
			$spedition->vat = (int) EshopOrdersConfig::load('speditions.vat', 21);

			$price = (float) ($priceLevel ? $priceLevel['price'] : $spedRaw['price']);
			if (EshopOrdersConfig::load('speditions.priceIsWithoutVat'))
				$price = round($price * (($spedition->vat / 100) + 1), 2);
			$spedition->setPrice($price);

			if ($priceLevel && $priceLevel['freeFrom'] !== null)
				$spedition->setFreeFrom((float) $priceLevel['freeFrom']);
			else if ($spedRaw['freeFrom'] !== null)
				$spedition->setFreeFrom((float) $spedRaw['freeFrom']);

			$spedition->isPickup       = $spedRaw['isPickup'] ? true : false;
			$spedition->allowOversized = $spedRaw['allowOversized'] ? true : false;
			$spedition->code1          = $spedRaw['code1'];
			$spedition->code2          = $spedRaw['code2'];
			$spedition->zboziId        = $spedRaw['zboziId'];
			$spedition->heurekaId      = $spedRaw['heurekaId'];
			$spedition->googleId       = $spedRaw['googleId'];

			$spedition->priceInBaseCurrency    = $spedition->getPrice();
			$spedition->freeFromInBaseCurrency = $spedition->getFreeFrom();

			$speditions[$spedition->getId()] = $spedition;
		}

		$this->eventDispatcher->dispatch(new PaymentsSpeditionsEvent(null, $speditions), Speditions::class . '::afterFillDao');

		return $speditions;
	}

	/**
	 * @param int $customerGroupId
	 *
	 * @return array|SpeditionPriceLevel[]
	 */
	protected function getGroupPriceLevels(int $customerGroupId): array
	{
		if ($this->cGroupPriceLevels === null) {
			$this->cGroupPriceLevels = [];
			foreach ($this->getEr()->findBy(['groupId' => $customerGroupId]) as $row) {
				/** @var SpeditionPriceLevel $row */
				$this->cGroupPriceLevels[$row->getSpedition()->getId()] = $row;
			}
		}

		return $this->cGroupPriceLevels;
	}
}

