<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Model\Subscribers;

use Core\Model\Event\ControlEvent;
use Core\Model\Event\EventDispatcher;
use Core\Model\Module;
use Core\Model\UI\AbstractPresenter;
use Core\Model\UI\FrontPresenter;
use EshopOrders\FrontModule\Components\Cart\CartPreview;
use EshopOrders\Model\Utils\Strings;
use Nette\Application\LinkGenerator;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\FrontModule\Model\Tags;
use EshopOrders\FrontModule\Model\Carts;
use EshopOrders\FrontModule\Model\Dao;
use EshopOrders\FrontModule\Model\Event\AddedCartItemEvent;
use EshopOrders\FrontModule\Model\Event\FillDaoItemsEvent;
use EshopOrders\FrontModule\Model\Event\RemovedCartItemEvent;
use EshopOrders\FrontModule\Model\Event\UpdatedCartItemEvent;
use EshopOrders\Model\Entities\CartItem;
use Nette\Localization\ITranslator;

class CartSubscriber implements EventSubscriberInterface
{
	/** @var Carts */
	protected $cartsService;

	/** @var ProductsFacade */
	protected $productsFacade;

	/** @var Tags */
	protected $tagsService;

	/** @var ITranslator */
	protected $translator;

	/** @var LinkGenerator */
	protected $linkGenerator;

	protected EventDispatcher $eventDispatcher;

	/** @var array */
	protected $cartChanges = [];

	protected $presenter;

	public function __construct(Carts $carts, ProductsFacade $productsFacade, Tags $tags, ITranslator $translator, LinkGenerator $linkGenerator,
	                            EventDispatcher $eventDispatcher)
	{
		$this->cartsService    = $carts;
		$this->productsFacade  = $productsFacade;
		$this->tagsService     = $tags;
		$this->translator      = $translator;
		$this->linkGenerator   = $linkGenerator;
		$this->eventDispatcher = $eventDispatcher;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			FrontPresenter::class . '::startup'      => 'frontPresenterStartup',
			FrontPresenter::class . '::beforeRender' => 'beforeRender',
			'eshopOrders.cartAddItem'                => ['onAddItem', 100],
			'eshopOrders.cartUpdateItem'             => ['onUpdateItem', 100],
			'eshopOrders.cartRemoveItem'             => ['onRemoveItem', 100],
			'eshopOrders.cartFillDaoItems'           => ['onFillDaoItems', 100],
		];
	}

	public function frontPresenterStartup(ControlEvent $event): void
	{
		/** @var AbstractPresenter $presenter */
		$presenter       = $event->control->getPresenter();
		$this->presenter = $presenter;

		$callback = function() use ($presenter): void {
			/** @var CartPreview $cartPreview */
			$cartPreview = $presenter->getComponent('cartPreview', false);
			if ($cartPreview) {
				$presenter->setPayloadAjaxPlaceholder('cartPreviewItemsCount', (string) $cartPreview->getItemsCountFormatted());
				$presenter->setPayloadAjaxPlaceholder('cartPreviewItemsCountRaw', (string) $cartPreview->getItemsCountRaw());
				$presenter->setPayloadAjaxPlaceholder('cartPreviewItemsPrice', (string) $cartPreview->getPriceFormatted());
			}

			$presenter->template->cart = $cartPreview->getCart();
			$presenter->redrawControl('cartMessage');
		};

		$this->eventDispatcher->addListener('eshopOrders.cartAddItem', $callback);
		$this->eventDispatcher->addListener('eshopOrders.cartUpdateItem', $callback);
		$this->eventDispatcher->addListener('eshopOrders.cartRemoveItem', $callback);
	}

	public function beforeRender(ControlEvent $event): void
	{
		if (!empty($this->cartChanges)) {
			$message = null;
			$event->control->redrawControl('orderCartDetail');

			if (isset($this->cartChanges['added']))
				$message = 'itemAdded';
			if (isset($this->cartChanges['updated']))
				$message = 'itemUpdated';
			if (isset($this->cartChanges['removed']))
				$message = 'itemRemoved';
			if ($message)
				$event->control->flashMessageSuccess('eshopOrdersFront.cart.' . $message);

			$event->control->redrawControl('flashes');
		}
	}

	public function onAddItem(AddedCartItemEvent $event): void
	{
		$item = &$event->item;

		$product = $this->productsFacade->getProduct($item->productId);
		if (!$item->ignoreValidation && (!$product || !$product->canAddToCart))
			return;

		$this->cartChanges['added'][] = $item;
		$this->cartsService->addItem($item);
	}

	public function onUpdateItem(UpdatedCartItemEvent $event): void
	{
		$this->cartChanges['updated'][] = $event->itemId;
		$this->cartsService->updateItemQuantity($event->itemId, $event->quantity, $event->moreData);
	}

	public function onRemoveItem(RemovedCartItemEvent $event): void
	{
		$this->cartChanges['removed'][] = $event->itemId;
		$this->cartsService->removeItem($event->itemId);
	}

	public function onFillDaoItems(FillDaoItemsEvent $event): void
	{
		$cartItems = $event->items;
		if ($this->cartsService->cDaoItems)
			return;

		$productIds = array_map(function(CartItem $ci) { return $ci->getProductId(); }, $cartItems);

		if (!Strings::startsWith(Module::$currentPresenterName, 'EshopOrders')) {
//			ProductsFacade::$loadFeatures = false;
			ProductsFacade::$loadTags     = false;
		}
		$products                     = $this->productsFacade->getProducts($productIds);
		ProductsFacade::$loadFeatures = true;
		ProductsFacade::$loadTags     = true;

		$items  = [];
		$childs = [];
		foreach ($cartItems as $ci) {
			$product = $products[$ci->getProductId()] ?? null;

			if (!$product)
				continue;

			if (!$ci->getMoreDataValue('isCustom', false) && (!$product || !$product->canAddToCart))
				continue;

			$cartItem = new Dao\CartItem((string) $product->getName(), $product->getId(), $product->getPrice());

			$cartItem
				->setId($ci->getId())
				->setProductId($ci->getProductId())
				->setProduct($product)
				->setQuantity($ci->quantity)
				->setLink($product->link)
				->setVatRate((int) $product->getVatRate());

			$cartItem->setData($ci->getMoreData());

			$cartItem->priceInBaseCurrency = $product->priceInBaseCurrency;
			$cartItem->discountDisabled    = $product->discountDisabled;

			if ($product->getGallery() && $product->getGallery()->getCover())
				$cartItem->setImage($product->getGallery()->getCover());

			$tagFree = $product->getTag('freeDelivery');
			if ($tagFree && !$tagFree->isAuto)
				$cartItem->freeDelivery = true;

			if ($ci->getParent())
				$childs[$ci->getParent()->getId()][$ci->getId()] = $cartItem;
			else
				$items[$ci->getId()] = $cartItem;
		}

		foreach ($childs as $parentId => $parentChilds) {
			foreach ($parentChilds as $childId => $child) {
				if (isset($items[$parentId]))
					$items[$parentId]->addChild($child);
			}
		}

		$this->cartsService->cDaoItems = $items;
	}
}
