<?php declare(strict_types = 1);

namespace EshopOrders\Model\Entities;

use Core\Model\Helpers\Traits\TExtraField;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;
use Core\Model\Entities\TId;
use EshopOrders\Model\EshopOrdersConfig;
use Users\Model\Entities\User;

/**
 * @ORM\Table("eshop_orders__customer")
 * @ORM\Entity
 */
class Customer
{
	use TId;
	use TExtraField;

	const EXTRA_FIELD_SECTION = 'eshopOrdersCustomer';

	/**
	 * @var Users\Model\Entities\User
	 * @ORM\OneToOne(targetEntity="Users\Model\Entities\User")
	 * @ORM\JoinColumn(name="user_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	public $user;

	/**
	 * @var string
	 * @ORM\Column(name="phone", type="string", nullable = true)
	 */
	protected $phone;

	/** defaultni (prave vybrana) dorucovaci adresa
	 * @var CustomerAddress
	 * @ORM\OneToOne(targetEntity="CustomerAddress")
	 * @ORM\JoinColumn(name="address_delivery", referencedColumnName="id", nullable = true, onDelete="SET NULL")
	 */
	public $addressDelivery;

	/** defaultni (prave vybrana) dorucovaci adresa
	 * @var CustomerAddress
	 * @ORM\OneToOne(targetEntity="CustomerAddress")
	 * @ORM\JoinColumn(name="address_invoice", referencedColumnName="id", nullable = true, onDelete="SET NULL")
	 */
	public $addressInvoice;

	/** vsechny adresy, ktere ma zakaznik ulozene. Daji se pouzit jako dorucovaci i fakturacni
	 * @var CustomerAddress[]
	 * @ORM\OneToMany(targetEntity="CustomerAddress", mappedBy="customer", indexBy="id")
	 */
	protected $addressesAvailable;

	/** Vytvorene objednavky
	 * @var ArrayCollection|Order[]
	 * @ORM\OneToMany(targetEntity="Order", mappedBy="customer", indexBy="id")
	 */
	protected $orders;

	/**
	 * @var GroupCustomers
	 * @ORM\ManyToOne(targetEntity="GroupCustomers", cascade={"persist"})
	 * @ORM\JoinColumn(name="id_group_customers", referencedColumnName="id", onDelete="SET NULL", nullable=true)
	 */
	protected $groupCustomers;

	/**
	 * @var Payment[]|ArrayCollection
	 * @ORM\ManyToMany(targetEntity="Payment", indexBy="id")
	 * @ORM\JoinTable(name="eshop_orders__customer_disabled_payment",
	 *     joinColumns={@ORM\JoinColumn(name="customer_id", referencedColumnName="id", onDelete="CASCADE")},
	 *     inverseJoinColumns={@ORM\JoinColumn(name="payment_id", referencedColumnName="id", onDelete="CASCADE")}
	 *     )
	 */
	public $disabledPayments;

	/**
	 * @var Spedition[]|ArrayCollection
	 * @ORM\ManyToMany(targetEntity="Spedition", indexBy="id")
	 * @ORM\JoinTable(name="eshop_orders__customer_disabled_spedition",
	 *     joinColumns={@ORM\JoinColumn(name="customer_id", referencedColumnName="id", onDelete="CASCADE")},
	 *     inverseJoinColumns={@ORM\JoinColumn(name="spedition_id", referencedColumnName="id", onDelete="CASCADE")}
	 *     )
	 */
	public $disabledSpeditions;

	public function __construct(User $user, $phone)
	{
		$this->user               = $user;
		$this->phone              = $phone;
		$this->addressesAvailable = new ArrayCollection();
		$this->orders             = new ArrayCollection();
		$this->disabledPayments   = new ArrayCollection();
		$this->disabledSpeditions = new ArrayCollection();
	}

	/**
	 * @return User
	 */
	public function getUser(): User
	{
		return $this->user;
	}

	/**
	 * @return string
	 */
	public function getPhone(): string
	{
		return $this->phone ?: "";
	}

	/**
	 * @param string $phone
	 *
	 * @return Customer
	 */
	public function setPhone(string $phone): Customer
	{
		$this->phone = $phone;

		return $this;
	}

	/**
	 * @return CustomerAddress
	 */
	public function getAddressDelivery()
	{
		return $this->addressDelivery;
	}

	/** kdyz nova adresa neni v seznamu dostupnych adres, tak ji pridame i tam
	 *
	 * @param CustomerAddress $addressDelivery
	 *
	 * @return Customer
	 */
	public function setAddressDelivery(CustomerAddress $addressDelivery): Customer
	{
		$this->addressDelivery = $addressDelivery;
		if (!$this->addressesAvailable->contains($addressDelivery)) { //TODO bude to takhle fungovat? A kdy nastavime teto adrese id_customer?
			$this->addressesAvailable->add($addressDelivery);
		}

		return $this;
	}

	/**
	 * @return CustomerAddress
	 */
	public function getAddressInvoice(): ?CustomerAddress
	{
		return $this->addressInvoice;
	}

	/** kdyz nova adresa neni v seznamu dostupnych adres, tak ji pridame i tam
	 *
	 * @param CustomerAddress $addressInvoice
	 *
	 * @return Customer
	 */
	public function setAddressInvoice(CustomerAddress $addressInvoice): Customer
	{
		$this->addressInvoice = $addressInvoice;
		if (!$this->addressesAvailable->contains($addressInvoice)) {
			$this->addressesAvailable->add($addressInvoice);
		}

		return $this;
	}

	/**
	 * @return CustomerAddress[]
	 */
	public function getAddressesAvailable(): array
	{
		return $this->addressesAvailable;
	}

	/**
	 * @param CustomerAddress[] $addressesAvailable
	 *
	 * @return Customer
	 */
	public function setAddressesAvailable(array $addressesAvailable): Customer
	{
		$this->addressesAvailable = $addressesAvailable;

		return $this;
	}

	/**
	 * @param CustomerAddress $address
	 *
	 * @return Customer
	 */
	public function addAddressAvailable($address): Customer
	{
		if (!$this->addressesAvailable->contains($address)) {
			$this->addressesAvailable->add($address);
		}

		return $this;
	}

	/**
	 * @return ArrayCollection|Order[]
	 */
	public function getOrders()
	{
		return $this->orders;
	}

	/** TODO pridavat a ubirat po jedne?
	 *
	 * @param ArrayCollection|Order[] $orders
	 *
	 * @return Customer
	 */
	public function setOrders($orders)
	{
		$this->orders = $orders;

		return $this;
	}

	/**
	 * @return GroupCustomers
	 */
	public function getGroupCustomers()
	{
		return $this->groupCustomers;
	}

	/**
	 * @param GroupCustomers $groupCustomers
	 */
	public function setGroupCustomers(GroupCustomers $groupCustomers)
	{
		$this->groupCustomers = $groupCustomers;
	}

	public function removeGroupCustomers()
	{
		$this->groupCustomers = null;
	}

	public function hasMinimalOrderPrice(): float
	{
		if (EshopOrdersConfig::load('customer.allowMinimalOrderPrice', false)) {
			$ef = $this->getExtraFieldsValues();

			if (isset($ef['allowMinimalOrderPrice']) && $ef['allowMinimalOrderPrice'] == true && $ef['minimalOrderPrice'] > 0) {
				return (float) str_replace(',', '.', (string) $ef['minimalOrderPrice']);
			}
		}

		return 0;
	}
}

