<?php declare(strict_types = 1);

namespace EshopOrders\Model\Entities;

use Core\Model\Helpers\Strings;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use Core\Model\Entities\TId;

/**
 * @ORM\Table("eshop_orders__spedition")
 * @ORM\Entity
 */
class Spedition
{
	use TId;

	/**
	 * @var string
	 * @ORM\Column(name="name", type="string")
	 */
	protected $name;

	/**
	 * @var string
	 * @ORM\Column(name="ident", type="string")
	 */
	protected $ident;

	/**
	 * @ORM\Column(name="code1", type="string", nullable=true)
	 */
	public ?string $code1 = null;

	/**
	 * @ORM\Column(name="code2", type="string", nullable=true)
	 */
	public ?string $code2 = null;

	/**
	 * @var int
	 * @ORM\Column(name="is_pickup", type="smallint", nullable=false, options={"default": 0, "unsigned": true})
	 */
	public int $isPickup = 0;

	/**
	 * @var int
	 * @ORM\Column(name="allow_oversized", type="smallint", nullable=false, options={"default": 0, "unsigned": true})
	 */
	public int $allowOversized = 0;

	/**
	 * @var string
	 * @ORM\Column(name="text", type="string")
	 */
	protected $text;

	/**
	 * @var int
	 * @ORM\Column(name="is_published", type="smallint")
	 */
	public $isPublished;

	/**
	 * @var int
	 * @Gedmo\SortablePosition
	 * @ORM\Column(name="position", type="integer")
	 */
	protected $position;

	/**
	 * @var float
	 * @ORM\Column(name="price", type="decimal", precision=10, scale=2)
	 */
	protected $price;

	/**
	 * @var int
	 * @ORM\Column(name="free_from", type="integer", nullable=true)
	 */
	protected $freeFrom;

	/**
	 * @var int
	 * @ORM\Column(name="available_from", type="integer", nullable=true)
	 */
	protected $availableFrom;

	/**
	 * @var int
	 * @ORM\Column(name="available_to", type="integer", nullable=true)
	 */
	protected $availableTo;

	/**
	 * @var string
	 * @ORM\Column(name="image", type="string", nullable=true, length=255)
	 */
	public $image;

	/**
	 * @ORM\Column(name="zbozi_id", type="string", nullable=true, length=255)
	 */
	public ?string $zboziId = null;

	/**
	 * @ORM\Column(name="heureka_id", type="string", nullable=true, length=255)
	 */
	public ?string $heurekaId = null;

	/**
	 * @var string|null
	 * @ORM\Column(name="google_id", type="string", nullable=true, length=255)
	 */
	public ?string $googleId = null;

	/**
	 * @var SpeditionPriceLevel[]|ArrayCollection
	 * @ORM\OneToMany(targetEntity="SpeditionPriceLevel", mappedBy="spedition", indexBy="group_id")
	 */
	protected $priceLevels;

	/**
	 * @var Customer[]|ArrayCollection
	 * @ORM\ManyToMany(targetEntity="Customer", mappedBy="disabledSpeditions", indexBy="id")
	 */
	protected $disabledForCustomers;

	/**
	 * Member constructor.
	 */
	public function __construct()
	{
		$this->isPublished = 1;
		$this->setPosition(-1);
		$this->freeFrom             = 999999;
		$this->priceLevels          = new ArrayCollection();
		$this->disabledForCustomers = new ArrayCollection();
	}

	public function getName()
	{
		return $this->name;
	}

	public function getText()
	{
		return $this->text;
	}

	public function getPosition()
	{
		return $this->position;
	}

	public function getPrice()
	{
		return $this->price;
	}

	public function setName($name)
	{
		$this->name = $name;

		return $this;
	}

	public function setText($text)
	{
		$this->text = $text;

		return $this;
	}

	public function setPosition($position)
	{
		$this->position = intval($position);

		return $this;
	}

	public function setPrice(float $price): self
	{
		$this->price = Strings::formatEntityDecimal($price);

		return $this;
	}

	/*******
	 * === Ident
	 */

	public function getIdent()
	{
		return $this->ident;
	}

	public function setIdent($ident): Spedition
	{
		$this->ident = $ident;

		return $this;
	}

	/*******
	 * === FreeFrom
	 */

	public function getFreeFrom(): ?int { return $this->freeFrom; }

	public function setFreeFrom(?int $freeFrom): Spedition
	{
		$this->freeFrom = $freeFrom;

		return $this;
	}

	/*******
	 * === AvailableFrom
	 */

	public function getAvailableFrom()
	{
		return $this->availableFrom;
	}

	public function setAvailableFrom(int $availableFrom): Spedition
	{
		$this->availableFrom = $availableFrom;

		return $this;
	}

	/*******
	 * === AvailableTo
	 */

	public function getAvailableTo()
	{
		return $this->availableTo;
	}

	public function setAvailableTo(int $availableTo): Spedition
	{
		$this->availableTo = $availableTo;

		return $this;
	}

	/**
	 * @return ArrayCollection|SpeditionPriceLevel[]
	 */
	public function getPriceLevels() { return $this->priceLevels; }

	/**
	 * @param SpeditionPriceLevel[] $priceLevels
	 */
	public function setPriceLevels(array $priceLevels)
	{
		$this->priceLevels = new ArrayCollection($priceLevels);
	}
}
