<?php declare(strict_types = 1);

namespace EshopOrders\Model;

use Core\Model\Countries;
use Core\Model\Helpers\BaseEntityService;
use Core\Model\Sites;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\ProductSpedition;
use EshopOrders\FrontModule\Model\Customers;
use EshopOrders\Model\Entities\PaymentSpedition;
use Doctrine\ORM\Query\Expr\Join;
use Users\Model\Security\User;
use EshopOrders\FrontModule\Model\Speditions;
use EshopOrders\FrontModule\Model\Payments;
use EshopOrders\FrontModule\Model\Dao;

/**
 * class PaymentSpeditions
 * @package EshopOrders\Model
 *
 * @method PaymentSpedition|object|null getReference($id)
 * @method PaymentSpedition|null get($id)
 */
class PaymentSpeditions extends BaseEntityService
{
	protected $entityClass = PaymentSpedition::class;

	public ?Dao\Cart $cart = null;

	/** @var Sites */
	public Sites $sitesService;

	/** @var Countries */
	protected Countries $countries;

	/** @var User */
	protected User $user;

	/** @var Customers */
	protected Customers $customers;

	/** @var Speditions */
	protected Speditions $speditions;

	/** @var Payments */
	protected Payments $payments;

	/** @var Dao\PaymentSpedition[]|null */
	protected ?array $cPublished = null;

	public function __construct(Sites $sitesService, User $user, Customers $customers, Countries $countries,
	                            Speditions $speditions, Payments $payments)
	{
		$this->sitesService = $sitesService;
		$this->countries    = $countries;
		$this->user         = $user;
		$this->customers    = $customers;
		$this->speditions   = $speditions;
		$this->payments     = $payments;
	}

	public function setPosition($id, $position)
	{
		if ($item = $this->get($id)) {
			$item->setPosition($position);
			$this->em->persist($item);
			$this->em->flush();

			return true;
		}

		return false;
	}

	public function setPublish($id, $state)
	{
		if ($item = $this->getReference($id)) {
			$item->isPublished = $state;
			$this->em->persist($item);
			$this->em->flush();

			return true;
		}

		return false;
	}

	/**
	 * @return Dao\PaymentSpedition[]
	 */
	public function getAllPublished(string $site = null): array
	{
		if ($site || $this->cPublished === null || EshopOrdersConfig::load('paymentSpeditions.frontDisableLoadingOptimalization', false)) {
			$this->cPublished   = [];
			$cart               = $this->cart;
			$payments           = $this->payments->getAllPublished();
			$speditions         = $this->speditions->getAllPublished();
			$disabledSpeditions = [];

			if ($cart && Config::load('product.allowModifySpedition')) {
				$productsIds = [];
				foreach ($cart->getCartItems() as $item)
					if ($item->getProductId())
						$productsIds[] = $item->getProductId();

				if ($productsIds) {
					foreach ($this->em->createQueryBuilder()
						         ->select('IDENTITY(ps.spedition) as sped')
						         ->from(ProductSpedition::class, 'ps')
						         ->where('ps.product IN (:ids)')
						         ->setParameter('ids', $productsIds)->getQuery()->getArrayResult() as $row) {
						$disabledSpeditions[$row['sped']] = true;
					}
				}
			}

			$qb = $this->getEr()->createQueryBuilder('ps')
				->select('ps.id as id, p.id as payment, s.id as spedition, GROUP_CONCAT(c.id) as countries, ps.freeFromDate, ps.freeToDate, ps.currency')
				->andWhere('ps.isPublished = :isPublished')
				->innerJoin('ps.countries', 'c')
				->innerJoin('ps.payment', 'p')
				->innerJoin('ps.spedition', 's')
				->addOrderBy('s.position', 'ASC')
				->addOrderBy('p.position', 'ASC')
				->setParameters([
					'isPublished' => 1,
				])
				->groupBy('ps.id');

			if (count($this->sitesService->getSites()) > 1) {
				$qb->innerJoin('ps.sites', 'sites', Join::WITH, 'sites.ident = :site')
					->setParameter('site', $site ?? $this->sitesService->getCurrentSite()->getIdent());
			}

			foreach ($qb->getQuery()->getArrayResult() as $ps) {
				$payment   = &$payments[$ps['payment']];
				$spedition = &$speditions[$ps['spedition']];

				if (!$payment || !$spedition || (Config::load('product.allowModifySpedition') && isset($disabledSpeditions[$ps['spedition']])))
					continue;

				$dao               = new Dao\PaymentSpedition((int) $ps['id'], $payment, $spedition);
				$dao->freeFromDate = $ps['freeFromDate'];
				$dao->freeToDate   = $ps['freeToDate'];
				$dao->currency     = $ps['currency'] ?: null;

				foreach (explode(',', $ps['countries']) as $v) {
					$country = $this->countries->getDao()[$v];
					if ($country)
						$dao->addCountry($country->getId(), $country);
				}

				$this->cPublished[$ps['id']] = $dao;
			}

			foreach ($this->cPublished as $id => &$dao) {
				if ($dao->isFreeCombination()) {
					$dao->getSpedition()->freeForPayments[] = &$payments[$dao->getPayment()->getId()];
					$dao->getPayment()->freeForSpeditions[] = &$speditions[$dao->getSpedition()->getId()];
				}
			}
		}

		return $this->cPublished;
	}

	public function getAllPublishedByCountry(?string $country = null): array
	{
		if (!$country)
			return $this->getAllPublished();

		$result = [];

		foreach ($this->getAllPublished() as $row) {
			if (isset($row->getCountries()[$country]))
				$result[$row->getId()] = $row;
		}

		return $result;
	}

	public function getFirstFreeSpedition(?string $country = null)
	{
		/** @var Dao\Spedition|null $min */
		$min = null;

		foreach ($this->getAllPublished() as $ps) {
			if (!$country)
				continue;

			$countries = $ps->getCountries()[$country] ?? $ps->getCountries()[strtolower($country)] ?? [];
			if (empty($countries))
				continue;

			if ($ps->isFreeCombination()) {
				$sped           = clone $ps->getSpedition();
				$sped->freeFrom = 0;
				$min            = $sped;
				break;
			}

			$spedition = $ps->getSpedition();
			if ($spedition->freeFrom !== null && $spedition->freeFrom > 0 && ($min === null || $spedition->freeFrom < $min->freeFrom))
				$min = $spedition;
		}

		return $min;
	}

	public function getFirstFreeSpeditionByType(?string $country = null)
	{
		/** @var Dao\Spedition|null $minStorePickup */
		$minStorePickup = null;

		/** @var Dao\Spedition|null $minPickup */
		$minPickup = null;

		/** @var Dao\Spedition|null $minAddress */
		$minAddress = null;

		foreach ($this->getAllPublished() as $ps) {
			$countries = array_change_key_case($ps->getCountries(), CASE_UPPER);
			$country   = $country ? strtoupper($country) : $country;
			if (!$country || ($country && !empty($countries) && !isset($countries[$country])))
				continue;

			if ($ps->isFreeCombination()) {
				$sped           = clone $ps->getSpedition();
				$sped->freeFrom = 0;

				if ($sped->isPickup && $sped->getIdent() === 'pickup') {
					$minStorePickup = $sped;
				} else if ($sped->isPickup) {
					$minPickup = $sped;
				} else {
					$minAddress = $sped;
				}

				break;
			}
			$spedition = $ps->getSpedition();

			if ($spedition->freeFrom !== null && ($minStorePickup === null || $spedition->freeFrom < $minStorePickup->freeFrom) && $spedition->isPickup && $spedition->getIdent() === 'pickup') {
				$minStorePickup = $spedition;
			} else if ($spedition->freeFrom !== null && $spedition->freeFrom > 0 && ($minPickup === null || $spedition->freeFrom < $minPickup->freeFrom) && $spedition->isPickup) {
				$minPickup = $spedition;
			} else if ($spedition->freeFrom !== null && $spedition->freeFrom > 0 && ($minAddress === null || $spedition->freeFrom < $minAddress->freeFrom) && !$spedition->isPickup) {
				$minAddress = $spedition;
			}
		}

		$result = [
			'storePickup' => $minStorePickup,
			'pickup'      => $minPickup,
			'address'     => $minAddress,
		];

		uasort($result, fn($sp1, $sp2) => $sp1->freeFrom <=> $sp2->freeFrom);

		return $result;
	}
}
