<?php declare(strict_types = 1);

namespace EshopOrders\Model;

use Core\Model\Helpers\BaseEntityService;
use Doctrine\ORM\Query;
use EshopOrders\Model\Entities\Payment;

/**
 * class Payments
 * @package EshopOrders\Model
 *
 * @method Payment|object|null getReference($id)
 * @method Payment|null get($id)
 */
class Payments extends BaseEntityService
{
	protected $entityClass = Payment::class;

	protected $cForSelectOption;

	public function setPosition($id, $position)
	{
		if ($item = $this->get($id)) {
			$item->setPosition($position);
			$this->em->persist($item);
			$this->em->flush();

			return true;
		}

		return false;
	}

	public function setPublish($id, $state)
	{
		if ($item = $this->getReference($id)) {
			$item->isPublished = $state;
			$this->em->persist($item);
			$this->em->flush();

			return true;
		}

		return false;
	}

	public function getAll()
	{
		$paymentsQuery = $this->getEr()->createQueryBuilder('p', 'p.id');
		$paymentsQuery->orderBy('p.position');
		$payments = $paymentsQuery->getQuery()->getResult();

		return $payments;
	}

	public function getAllPublished()
	{
		$paymentsQuery = $this->getEr()->createQueryBuilder('p', 'p.id');
		$paymentsQuery->andWhere('p.isPublished = :isPublished')->setParameter('isPublished', true);
		$paymentsQuery->orderBy('p.position');
		$payments = $paymentsQuery->getQuery()->getResult();

		return $payments;
	}

	/**
	 * @param int $cartValue cena objednavky
	 *
	 * @return Dao\Payment[] vsechny dopravy, ktere jsou aktivni, a jsou dostupne pro danou cenu objednavky
	 */
	public function getAllByCartValue($cartValue)
	{
		$paymentsQuery = $this->getEr()->createQueryBuilder('p', 'p.id');
		$paymentsQuery->andWhere('p.isPublished = :isPublished')->setParameter('isPublished', true);
		$paymentsQuery->andWhere('p.availableFrom IS NULL OR p.availableFrom <= :cartValue')->setParameter('cartValue', $cartValue);
		$paymentsQuery->andWhere('p.availableTo IS NULL OR p.availableTo >= :cartValue')->setParameter('cartValue', $cartValue);
		$paymentsQuery->orderBy('p.position');
		$payments = $paymentsQuery->getQuery()->getResult();

		return $payments;
	}

	/**
	 * @param      $ident
	 * @param bool $onlyActive
	 * @param int  $hydrate
	 *
	 * @return array|null|Payment
	 */
	public function getByIdent($ident, bool $onlyActive = false, int $hydrate = Query::HYDRATE_ARRAY)
	{
		$qb = $this->getEr()->createQueryBuilder('p');
		$qb->where('p.ident = :ident')
			->setParameter('ident', $ident)
			->setMaxResults(1);

		if ($onlyActive) {
			$qb->andWhere('p.isPublished = 1');
		}

		return $qb->getQuery()->getOneOrNullResult($hydrate);
	}

	public function getReferenceByIdent($ident)
	{
		return $this->getReference($this->getByIdent($ident)['id']);
	}

	public function getForSelectOption(): array
	{
		if ($this->cForSelectOption === null)
			$this->cForSelectOption = [null => ''] + ($this->getEr()->findPairs([], 'name') ?: []);

		return $this->cForSelectOption;
	}
}
