<?php declare(strict_types = 1);

namespace EshopOrders\Model\Utils;

use Core\Model\Dao\Site;
use Core\Model\Templating\Filters\Price;
use EshopCatalog\FrontModule\Model\Dao\Seller;
use EshopOrders\Model\Entities\IDiscount;
use EshopOrders\Model\Entities\OrderDiscount;
use EshopOrders\Model\Entities\OrderItem;
use EshopOrders\Model\EshopOrdersConfig;

class Helpers
{
	/** @var Price */
	protected $priceFilter;

	/**
	 * Helpers constructor.
	 *
	 * @param Price $priceFilter
	 */
	public function __construct(Price $priceFilter)
	{
		$this->priceFilter = $priceFilter;
	}

	/**
	 * @param string $input
	 * @param array  $timeWildcards
	 *
	 * @return string
	 */
	public static function replaceTimeWildcards(string $input, array $timeWildcards): string
	{
		foreach ($timeWildcards as $item) {
			switch ($item) {
				case 'YYYY':
					$input = str_replace('YYYY', date('Y'), $input);
					break;
				case 'YY':
					$input = str_replace('YY', date('y'), $input);
					break;
				case 'MM':
					$input = str_replace('MM', date('m'), $input);
					break;
			}
		}

		return $input;
	}

	/**
	 * @param IDiscount $discount
	 *
	 * @return string
	 */
	public function formatDiscount(IDiscount $discount): string
	{
		if ($discount->getType() === OrderDiscount::TYPE_FIX) {
			return $this->priceFilter->format($discount->getValue());
		}

		// type percent
		return sprintf('%s %s', $discount->getValue(), '%');
	}

	/**
	 * @param OrderItem[] $orderItems
	 *
	 * @return bool
	 */
	public static function isCorrectiveByOrderItems(array $orderItems): bool
	{
		foreach ($orderItems as $orderItem) {
			if ($orderItem->getPrice() < 0) {
				return true;
			}
		}

		return false;
	}

	/**
	 * @param Seller $seller
	 * @param Site|\Core\Model\Entities\Site       $site
	 *
	 * @return string
	 */
	public static function getSellerEmail(Seller $seller, $site): string
	{
		if ($seller->sendSellerEmailType === \EshopCatalog\Model\Entities\Seller::EMAIL_TYPE_SELLER)
			return EshopOrdersConfig::load("orderForm.eshopSellerOtherEmail.{$site->getIdent()}.email", null) ?: $seller->email;

		return $site->email;
	}

	public static function getSellerName(Seller $seller, string $siteIdent): string
	{
		return EshopOrdersConfig::load("orderForm.eshopSellerOtherEmail.{$siteIdent}.name", null) ?: $seller->name;
	}
}
