<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Customer;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\Model\Config;
use EshopOrders\AdminModule\Model\GroupsCustomers;
use EshopOrders\Model\Helpers\CustomersCache;
use EshopOrders\Model\Entities\GroupCustomers;
use EshopOrders\Model\Entities\GroupParam;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Helpers\CustomersCacheHelper;
use MeasuringCodes\DI\MeasuringCodesExtension;
use Nette\InvalidArgumentException;
use Nette\Utils\ArrayHash;

class GroupCustomersForm extends BaseControl
{
	public ?GroupCustomers $groupCustomers = null;

	protected GroupsCustomers $groupsCustomersService;
	protected CustomersCache  $customersCache;

	public function __construct(
		GroupsCustomers $groupsCustomers,
		CustomersCache  $customersCache
	)
	{
		$this->groupsCustomersService = $groupsCustomers;
		$this->customersCache         = $customersCache;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();

		$form->addText('short', 'eshopOrders.groupCustomersForm.short')->setMaxLength(60);
		$form->addText('name', 'eshopOrders.groupCustomersForm.name')->setRequired()->setMaxLength(255);
		$form->addBool('isPublished', 'eshopOrders.groupCustomersForm.isPublished')->setDefaultValue(0);

		if (Config::load('enablePriceLevels', false)) {
			$form->addText('productsSale', 'eshopOrders.groupCustomer.productsSale')
				->setHtmlType('number')
				->setHtmlAttribute('min', 0)
				->setHtmlAttribute('step', .01)
				->setDescription('eshopOrders.groupCustomer.productsSaleDescription')
				->setDefaultValue(0);

			if (EshopOrdersConfig::load('customerGroup.allowUseSaleOnOtherGroupPrices')) {
				$form->addSelect(
					'useProductsSaleOnGroup',
					'eshopOrders.groupCustomer.useProductsSaleOnGroup',
					['' => ''] + $this->groupsCustomersService->getOptionsForSelect(),
				)
					->setDescription('eshopOrders.groupCustomer.useProductsSaleOnGroupDescription');
			}

			if (EshopOrdersConfig::load('customerGroup.allowChangeProductSaleBasePrice')) {
				$form->addBool('productsSaleUseGroupPrice', 'eshopOrders.groupCustomer.productsSaleUseGroupPrice')
					->setDefaultValue(0);
			}

			if (EshopOrdersConfig::load('customerGroup.allowInheritGroupPrices')) {
				$form->addSelect(
					'inheritGroupPrices',
					'eshopOrders.groupCustomer.inheritGroupPrices',
					['' => ''] + $this->groupsCustomersService->getOptionsForSelect(),
				);
			}
		}

		$paramsContainer = $form->addContainer('params');
		if (EshopOrdersConfig::load('customerGroup.enableRequireIco')) {
			$paramsContainer->addBool('requireIco', 'eshopOrders.groupCustomer.requireIco')
				->setDefaultValue(0);
		}

		if (class_exists(MeasuringCodesExtension::class) && EshopOrdersConfig::load('customerGroup.allowDisableMeasuringCodes')) {
			$form->addBool('disableMeasuringCodes', 'eshopOrders.groupCustomer.disableMeasuringCodes')
				->setDefaultValue(0);
		}

		$form->addSaveCancelControl('saveControl');

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			$groupCustomers = $this->groupCustomers ?: new GroupCustomers($values->short, $values->name);

			$groupCustomers->short                     = $values->short;
			$groupCustomers->name                      = $values->name;
			$groupCustomers->isPublished               = $values->isPublished;
			$groupCustomers->useProductsSaleOnGroup    = EshopOrdersConfig::load('customerGroup.allowUseSaleOnOtherGroupPrices') && $values->useProductsSaleOnGroup
				? $this->groupsCustomersService->getReference($values->useProductsSaleOnGroup)
				: null;
			$groupCustomers->productsSaleUseGroupPrice = EshopOrdersConfig::load('customerGroup.allowChangeProductSaleBasePrice')
				? (int) $values->productsSaleUseGroupPrice
				: 0;
			$groupCustomers->inheritPricesGroup        = EshopOrdersConfig::load('customerGroup.allowInheritGroupPrices') && $values->inheritGroupPrices
				? $this->groupsCustomersService->getReference($values->inheritGroupPrices)
				: null;

			$groupCustomers->setProductsSale((float) $values->productsSale);

			if (class_exists(MeasuringCodesExtension::class) && EshopOrdersConfig::load('customerGroup.allowDisableMeasuringCodes')) {
				$groupCustomers->disableMeasuringCodes = (int) $values->disableMeasuringCodes;
			}

			/** @var GroupParam[] $params */
			$params = $groupCustomers->getParams()->toArray();

			foreach ($values->params as $k => $v) {
				if (isset($params[$k])) {
					$params[$k]->value = (string) $v;
					$this->em->persist($params[$k]);
				} else {
					$param        = new GroupParam($groupCustomers, $k);
					$param->value = (string) $v;
					$this->em->persist($param);
				}
			}

			$this->em->persist($groupCustomers);
			$this->em->flush();
			$form->addCustomData('groupCustomersId', $groupCustomers->getId());

			$this->customersCache->getCache()->remove(CustomersCacheHelper::getCustomerGroupKey($groupCustomers->getId()));

			$this->getPresenter()->flashMessageSuccess('eshopOrders.groupCustomersForm.groupCustomersSaved');
		} catch (\Exception $e) {
			$form->addError($e->getMessage());
			$this->redrawControl('form');
		}

		return false;
	}

	public function setGroupCustomers($id): void
	{
		$this->groupCustomers = $this->em->getRepository(GroupCustomers::class)->find($id);

		if (!$this->groupCustomers)
			throw new InvalidArgumentException();

		$d = [
			'short'       => $this->groupCustomers->short,
			'name'        => $this->groupCustomers->name,
			'isPublished' => $this->groupCustomers->isPublished,
		];

		if (class_exists(MeasuringCodesExtension::class) && EshopOrdersConfig::load('customerGroup.allowDisableMeasuringCodes')) {
			$d['disableMeasuringCodes'] = $this->groupCustomers->disableMeasuringCodes;
		}

		if (Config::load('enablePriceLevels', false)) {
			$d['productsSale'] = str_replace(',', '.', (string) $this->groupCustomers->getProductsSale());

			if (EshopOrdersConfig::load('customerGroup.allowUseSaleOnOtherGroupPrices')) {
				$d['useProductsSaleOnGroup'] = $this->groupCustomers->useProductsSaleOnGroup ? $this->groupCustomers->useProductsSaleOnGroup->getId() : null;
			}

			if (EshopOrdersConfig::load('customerGroup.allowChangeProductSaleBasePrice')) {
				$d['productsSaleUseGroupPrice'] = $this->groupCustomers->productsSaleUseGroupPrice;
			}

			if (EshopOrdersConfig::load('customerGroup.allowInheritGroupPrices')) {
				$d['inheritGroupPrices'] = $this->groupCustomers->inheritPricesGroup ? $this->groupCustomers->inheritPricesGroup->getId() : null;
			}
		}

		foreach ($this->groupCustomers->getParams() as $k => $v) {
			$d['params'][$k] = $v->value;
		}

		$this['form']->setDefaults($d);
	}
}
