<?php declare(strict_types = 1);

namespace EshopOrders\ApiModule\Api\V1\Controllers;

use Apitte\Core\Annotation\Controller\Method;
use Apitte\Core\Annotation\Controller\Path;
use Apitte\Core\Http\ApiRequest;
use Apitte\Core\Http\ApiResponse;
use Apitte\Negotiation\Http\ArrayEntity;
use Apitte\Core\Annotation\Controller\RequestParameters;
use Core\Model\Application\AppState;
use Apitte\Core\Annotation\Controller\RequestParameter;
use Core\Model\Event\EventDispatcher;
use Core\Model\Entities\Site;
use Core\Model\Sites;
use EshopOrders\ApiModule\Api\V1\Exceptions\ErrorException;
use EshopOrders\ApiModule\Api\V1\Model\Orders;
use EshopOrders\FrontModule\Model\Customers;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use EshopOrders\Model\Entities\InvoiceConfig;
use EshopOrders\Model\Entities\OrderAddress;
use EshopOrders\Model\Entities\OrderDiscount;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Event\OrderPayloadReturnEvent;
use EshopOrders\Model\Invoices;
use EshopOrders\Model\Receipts;
use EshopOrders\Model\Speditions;
use EshopOrders\Model\Statuses;
use EshopOrders\Model\Payments;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderPayment;
use EshopOrders\Model\Entities\OrderSpedition;
use EshopOrders\Model\Entities\OrderStatus;
use Core\Model\Entities\EntityManagerDecorator;
use EshopOrders\Model\Utils\Helpers;
use EshopStock\Model\Repository\SupplyProducts;
use Apitte\Core\Annotation\Controller\RequestBody;
use Nette\Utils\DateTime;
use Nette\Utils\Validators;
use Tracy\Debugger;


/**
 * @Path("/orders")
 */
class OrdersController extends BaseController
{
	protected EventDispatcher        $eventDispatcher;
	protected EntityManagerDecorator $em;
	protected Sites                  $sitesService;
	protected Payments               $paymentsService;
	protected Speditions             $speditionsService;
	protected Statuses               $statusesService;
	protected Customers              $customersService;
	protected Invoices               $invoices;
	protected Orders                 $orders;
	protected Receipts               $receipts;

	public function __construct(
		EventDispatcher        $eventDispatcher,
		EntityManagerDecorator $em,
		Sites                  $sitesService,
		Payments               $paymentsService,
		Speditions             $speditionsService,
		Statuses               $statusesService,
		Customers              $customersService,
		Invoices               $invoices,
		Orders                 $orders,
		Receipts               $receipts
	)
	{
		$this->em                = $em;
		$this->sitesService      = $sitesService;
		$this->paymentsService   = $paymentsService;
		$this->speditionsService = $speditionsService;
		$this->statusesService   = $statusesService;
		$this->customersService  = $customersService;
		$this->eventDispatcher   = $eventDispatcher;
		$this->invoices          = $invoices;
		$this->orders            = $orders;
		$this->receipts          = $receipts;
	}

	/**
	 * @Path("/")
	 * @Method("GET")
	 */
	public function index(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		return $response->withStatus(ApiResponse::S200_OK)->withEntity(ArrayEntity::from($this->orders->getStatusesOverview()));
	}

	/**
	 * @Path("/{id}")
	 * @Method("GET")
	 * @RequestParameters({
	 *      @RequestParameter(name="id", type="int", required=true),
	 * })
	 */
	public function getDetail(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		try {
			return $response->withStatus(ApiResponse::S200_OK)->withEntity(ArrayEntity::from(['order' => $this->orders->getDetail($request->getParameter('id'))]));
		} catch (ErrorException $ex) {
			return $this->sendError($response, 'orderNotFound');
		}
	}

	/**
	 * @Path("/status/{status}")
	 * @Method("GET")
	 * @RequestParameters({
	 *      @RequestParameter(name="status", type="string", required=true),
	 * })
	 */
	public function getByStatus(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		return $response->withStatus(ApiResponse::S200_OK)->withEntity(ArrayEntity::from($this->orders->getIdByStatus($request->getParameter('status'))));
	}

	/**
	 * @Path("/status-not/{status}")
	 * @Method("GET")
	 * @RequestParameters({
	 *      @RequestParameter(name="status", type="string", required=true),
	 * })
	 */
	public function getByStatusNot(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		return $response->withStatus(ApiResponse::S200_OK)->withEntity(ArrayEntity::from($this->orders->getIdByStatusNot($request->getParameter('status'))));
	}

	/**
	 * @Path("/{id}/status")
	 * @Method("POST")
	 * @RequestParameters({
	 *      @RequestParameter(name="id", type="int", required=true),
	 * })
	 * @RequestBody(entity="EshopOrders\ApiModule\Api\V1\BodyEntity\OrderStatus")
	 */
	public function setOrderStatus(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		return $response->withStatus(ApiResponse::S200_OK)
			->withEntity(ArrayEntity::from([
				'result' => $this->orders->setOrderStatus(
					$request->getParameter('id'),
					$request->getEntity(),
				),
			]));
	}

	/**
	 * @Path("/{id}/payment")
	 * @Method("POST")
	 * @RequestParameters({
	 *      @RequestParameter(name="id", type="int", required=true),
	 * })
	 * @RequestBody(entity="EshopOrders\ApiModule\Api\V1\BodyEntity\Payment")
	 */
	public function changePayment(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		return $response->withStatus(ApiResponse::S200_OK)
			->withEntity(ArrayEntity::from([
				'result' => $this->orders->changePayment(
						$request->getParameter('id'),
						$request->getEntity(),
					) !== null,
			]));
	}

	/**
	 * @Path("/{id}/spedition")
	 * @Method("POST")
	 * @RequestParameters({
	 *      @RequestParameter(name="id", type="int", required=true),
	 * })
	 * @RequestBody(entity="EshopOrders\ApiModule\Api\V1\BodyEntity\Spedition")
	 */
	public function changeSpedition(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		return $response->withStatus(ApiResponse::S200_OK)
			->withEntity(ArrayEntity::from([
				'result' => $this->orders->changeSpedition(
						$request->getParameter('id'),
						$request->getEntity(),
					) !== null,
			]));
	}

	/**
	 * @Path("/create")
	 * @Method("POST")
	 */
	public function create(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		$data = $request->getJsonBody();

		try {
			$this->em->getConnection()->beginTransaction();

			$order = null;
			if (!Validators::isNone($data['orderId'])) {
				$order = $this->em->getRepository(Order::class)->find((int) $data['orderId']);
			}

			$order = $order ?? new Order($this->em->getReference(Site::class, $this->sitesService->getCurrentSite()->getIdent()));

			$event                 = new OrderEvent($order);
			$event->orderPayment   = $order->getPayment();
			$event->orderSpedition = $order->getSpedition();
			$event->addrDelivery   = $order->getAddressDelivery();
			$event->addrInvoice    = $order->getAddressInvoice();
			$event->orderItems     = $order->getOrderItems()->toArray();
			$event->formData       = $data;
			$this->eventDispatcher->dispatch($event, 'eshopOrders.admin.beforeOrderChange');

			$order->setMessage($data['globalNote'] ?? '');
			$order->setAgreedTerms(true);

			AppState::setState('eshopOrdersPaymentMethod', 'eshopOrders.paymentMethod.checkout');
			$order->isPaid = 1;
			$order->paid   = new DateTime;
			$orderItems    = $this->orders->convertJSONProductsToOrderItems($data['checkoutProducts'], $order);
			$isCorrective  = Helpers::isCorrectiveByOrderItems($orderItems);
			$order->setOrderItems($orderItems);
			$order->isCorrectiveTaxDocument = (int) $isCorrective;

			$payment = $this->paymentsService->getReferenceByIdent($data['payment']);
			$orderPayment = $order->getPayment() ?? new OrderPayment($payment, $order);
			$orderPayment->setName($payment->getName());
			$orderPayment->setPrice((float) $payment->getPrice());
			$orderPayment->setPayment($payment);

			$spedition = $this->speditionsService->getReferenceByIdent($data['spedition']);
			$orderSpedition = $order->getSpedition() ?? new OrderSpedition($spedition, $order);
			$orderSpedition->setName($spedition->getName());
			$orderSpedition->setPrice((float) $spedition->getPrice());
			$orderSpedition->setSpedition($spedition);

			$orderStatuses = [];
			$hasStatusCreated = false;
			$hasStatusSpedition = false;
			$hasStatusFinished = false;
			if ($order->getId() !== null) {
				foreach ($order->getOrderStatuses()->toArray() as $orderStatus) {
					if ($orderStatus->getStatus()->getId() === OrderStatus::STATUS_CREATED) {
						$hasStatusCreated = true;
					} else if ($orderStatus->getStatus()->getId() === OrderStatus::STATUS_SPEDITION) {
						$hasStatusSpedition = true;
					} else if ($orderStatus->getStatus()->getId() === OrderStatus::STATUS_FINISHED) {
						$hasStatusFinished = true;
					}
				}
			}

			$orderActionCreated = null;
			if (!$hasStatusCreated) {
				$statusCreated        = $this->statusesService->get(OrderStatus::STATUS_CREATED);
				$orderActionCreated   = new OrderStatus($order, $statusCreated);
				$orderStatuses[] = $orderActionCreated;
			}

			$orderActionSpedition = null;
			if (!$hasStatusSpedition) {
				$statusSpedition      = $this->statusesService->get(OrderStatus::STATUS_SPEDITION);
				$orderActionSpedition = new OrderStatus($order, $statusSpedition);
			}

			$orderActionFinished = null;
			if (!$hasStatusFinished) {
				$statusFinished       = $this->statusesService->get(OrderStatus::STATUS_FINISHED);
				$orderActionFinished  = new OrderStatus($order, $statusFinished);
				$orderStatuses[] = $orderActionFinished;
			}

			$order->setPayment($orderPayment);
			$order->setSpedition($orderSpedition);

			if ($order->getId() === null && count($orderStatuses) > 0) {
				$order->setOrderStatuses($orderStatuses);
			}

			$addressInvoice = $order->getAddressInvoice();
			$addressDelivery = $order->getAddressDelivery();
			if ($order->getId() === null) {
				$addressInvoice  = $this->orders->createAddress(new OrderAddress(OrderAddress::ADDRESS_INVOICE), $data['customer']['addressInvoice']);
				$addressDelivery = $this->orders->createAddress(new OrderAddress(OrderAddress::ADDRESS_DELIVERY), $data['customer']['addressDelivery']);
				$order->setAddressInvoice($addressInvoice);
				$order->setAddressDelivery($addressDelivery);

				if ($data['customer']['customerId'] && $customer = $this->customersService->get((int) $data['customer']['customerId'])) {
					$order->setCustomer($customer);
					$this->em->persist($customer);
				}
			}

			$removedOrderDiscountIds = array_diff(
				array_map(static fn(OrderDiscount $oi) => $oi->getId(), $order->getOrderDiscounts()->toArray()),
				array_filter(array_map(static fn(array $data) => $data['id'], $data['globalSales']), static fn($val) => !Validators::isNone($val))
			);
			foreach ($removedOrderDiscountIds as $discountId) {
				$orderDiscount = $this->em->getReference(OrderDiscount::class, (int) $discountId);
				if ($orderDiscount) {
					$this->em->remove($orderDiscount);
				}
			}
			$this->orders->addGlobalSales($data['globalSales'], $order, $data['totalPrice']);

			/*if ($data['customer']['customerCreated'] && ($data['customer']['delivery']['email'] || $data['customer']['invoice']['email'])) {
				// Id vkladam jenom formalne - nemel by existovat
				$customer = $this->customersService->getOrCreateCustomer($data['customer']['customerId']);
				$customer->setAddressDelivery($this->createAddress(new CustomerAddress($customer), $data['customer']['delivery']));
				$customer->setAddressInvoice($this->createAddress(new CustomerAddress($customer), $data['customer']['delivery']));
				$this->em->persist($customer);
			}*/

			$event                 = new OrderEvent($order);
			$event->orderPayment   = $orderPayment;
			$event->orderSpedition = $orderSpedition;
			$event->addrDelivery   = $addressDelivery;
			$event->addrInvoice    = $addressInvoice;
			$event->orderItems     = $orderItems;
			$event->formData       = $data;

			$eshopStockSupplyProduct = class_exists('EshopStock\DI\EshopStockExtension') ? SupplyProducts::class : null;

			if ($eshopStockSupplyProduct !== null) {
				$eshopStockSupplyProduct::$assignOrderItemFlush = false;
			}

			$this->eventDispatcher->dispatch($event, 'eshopCheckout.orderBeforeSave');

			if (EshopOrdersConfig::load('receipt.enableUseNumericalSeries', false) && in_array($orderPayment->getIdent(), ['storeCard', 'storeCash'])) {
				$order->receiptIdent = $this->receipts->generateIdent($order);
			}

			$this->em->persist($order);
			$this->em->persist($orderSpedition);
			$this->em->persist($orderPayment);
			if ($orderActionCreated) {
				$this->em->persist($orderActionCreated);
			}
			if ($orderActionSpedition) {
				$this->em->persist($orderActionSpedition);
			}
			if ($orderActionFinished) {
				$this->em->persist($orderActionFinished);
			}

			$this->em->flush();

			if ($isCorrective) {
				$invoice = $this->invoices->getOneByOrder($order, true, InvoiceConfig::TYPE_CORRECTIVE);
				$order->setInvoice($invoice);
				$this->em->flush();
				$this->eventDispatcher->dispatch($event, 'eshopOrders.admin.correctiveTaxDocumentCreated');
			}

			$this->em->getConnection()->commit();

			$this->eventDispatcher->dispatch($event, 'eshopOrders.admin.orderOnSuccess');

			$result = [
				'success' => $order,
			];

			$orderPayloadReturnEvent = new OrderPayloadReturnEvent($result, $order);
			$this->eventDispatcher->dispatch($orderPayloadReturnEvent, 'eshopOrders.api.orderPayloadReturnEvent');

			return $response->withStatus(ApiResponse::S200_OK)->withEntity(ArrayEntity::from($orderPayloadReturnEvent->data));
		} catch (\Exception $e) {
			Debugger::log($e);
			if ($this->em->getConnection()->isTransactionActive()) {
				$this->em->getConnection()->rollBack();
			}

			return $response->withStatus(ApiResponse::S200_OK)->withEntity(ArrayEntity::from([
				'error' => $e->getMessage(),
				'data'  => $data,
			]));
		}
	}
}
