<?php declare(strict_types = 1);

namespace EshopOrders\ApiModule\Api\V1\Model;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Helpers\Arrays;
use EshopOrders\ApiModule\Api\V1\Model\Dao;
use EshopOrders\Model\Entities\GroupCustomers;
use EshopOrders\Model\Entities\GroupParam;
use Tracy\Debugger;

class CustomerGroups
{
	protected EntityManagerDecorator $em;

	public function __construct(
		EntityManagerDecorator $em
	)
	{
		$this->em = $em;
	}

	/**
	 * @return Dao\CustomerGroup[]
	 */
	public function getAll(): array
	{
		$result = [];

		foreach ($this->em->getRepository(GroupCustomers::class)->createQueryBuilder('cg')
			         ->select('cg.id, cg.short, cg.name, cg.isPublished, cg.productsSale as sale')
			         ->getQuery()->getArrayResult() as $row) {
			$row['sale'] = (float) $row['sale'];

			$dao = new Dao\CustomerGroup($row['id'], $row['short'], $row['name']);
			Arrays::setObjectVars($dao, $row);

			$result[$row['id']] = $dao;
		}

		return $result;
	}

	public function setParameter(int $groupId, string $key, string $value): ?array
	{
		try {
			/** @var GroupParam|null $param */
			$param = $this->em->getRepository(GroupParam::class)->findOneBy([
				'group' => $groupId,
				'key'   => $key,
			]);

			if (!$param) {
				$param = new GroupParam(
					$this->em->getReference(GroupCustomers::class, $groupId),
					$key,
				);
			}

			$param->value = $value;

			$this->em->persist($param);
			$this->em->flush($param);

			return [
				'groupId' => $param->group->getId(),
				'key'     => $param->key,
				'value'   => $param->value,
			];
		} catch (\Exception $e) {
			Debugger::log('Set group ' . $groupId . ' parameter ' . $key . ' to value ' . $value . ' : ' . $e->getMessage(), 'api-eshoporders-customer-groups');
			Debugger::log($e, 'api-eshoporders-customer-groups');
		}

		return null;
	}
}
