<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Components\OrderItemReviews;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopOrders\Model\Entities\OrderItem;
use EshopOrders\Model\Entities\OrderItemReview;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Helpers\EshopOrdersCache;
use Nette\Caching\Cache;
use Nette\Utils\ArrayHash;
use Nette\Utils\DateTime;

class OrderItemReviewForm extends BaseControl
{
	protected OrderItem $orderItem;
	protected EshopOrdersCache $eshopOrdersCache;

	public function __construct(OrderItem $orderItem, EshopOrdersCache $eshopOrdersCache)
	{
		$this->orderItem = $orderItem;
		$this->eshopOrdersCache = $eshopOrdersCache;
	}

	protected function attached($presenter): void
	{
		parent::attached($presenter);

		$orderItemReview = $this->orderItem->orderItemReview->first();

		if (!$orderItemReview) {
			return;
		}

		$this['form']->setDefaults([
			'description' => $orderItemReview->description,
			'rating'      => (int) $orderItemReview->rating,
		]);
	}

	public function render(): void
	{
		$this->template->orderItem = $this->orderItem;
		$this->template->render($this->getTemplateFile());
	}

	public function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax()->setShowLangSwitcher(false);

		$form->addTextArea('description', 'eshopOrdersFront.orderItemReviewForm.description')
			 ->setRequired();
		$form->addText('customer', 'eshopOrdersFront.orderItemReviewForm.customer')
			 ->setNullable();
		$form->addRadioList('rating', null, array_fill_keys(range(1, EshopOrdersConfig::load('orderItemReviews.maxRating'), 1), null));

		$form->addSubmit('submit', 'eshopOrdersFront.orderItemReviewForm.addReview');

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): void
	{
		$orderItem = $this->orderItem;
		$orderItemReview = $orderItem->orderItemReview->first();

		if (!$orderItemReview) {
			$orderItemReview = new OrderItemReview($orderItem, new DateTime, (float) $values->rating, $values->description);
		}

		$orderItem->orderItemReview->add($orderItemReview);
		$orderItemReview->rating = (float) $values->rating;
		$orderItemReview->description = $values->description;
		$orderItemReview->customer = $values->customer;

		$this->em->persist($orderItem);
		$this->em->persist($orderItemReview);
		$this->em->flush();
		
		$this->eshopOrdersCache->getCache()->clean([
			Cache::Tags => ['productReviews/' . $orderItem->getProductId()],
		]);

		$this->presenter->flashMessageSuccess('default.saved');

		$this->redrawControl('form');
	}

}
