<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Model\Dao;

use EshopOrders\Model\EshopOrdersConfig;
use Nette\SmartObject;

class BasePS
{
	use SmartObject;

	/** @var int */
	public $id;

	/** @var string */
	public $name;

	/** @var string */
	public $ident;

	/** @var string */
	public $text;

	/** @var int */
	public $isPublished;

	/** @var int */
	public $position;

	/** @var float */
	public $price;

	public int $vat = 0;

	/** @var float */
	public $priceInBaseCurrency;

	/** @var float */
	public $freeFrom = null;

	/** @var float */
	public $freeFromInBaseCurrency = null;

	/** @var int */
	public $availableFrom;

	/** @var int */
	public $availableTo;

	/** @var string */
	public $image;

	/** @var Cart */
	public $cart;

	public ?string $country = null;

	public function getId(): int { return $this->id; }

	public function setId(int $id): self
	{
		$this->id = $id;

		return $this;
	}

	public function getIdent(): ?string { return $this->ident; }

	public function setIdent($ident): self
	{
		$this->ident = $ident;

		return $this;
	}

	public function getName(): string { return $this->name; }

	public function setName(string $name): self
	{
		$this->name = $name;

		return $this;
	}

	public function getText(): ?string { return $this->text; }

	public function setText(string $text): self
	{
		$this->text = $text;

		return $this;
	}

	public function isPublished(): bool { return (bool) $this->isPublished; }

	public function setPublished($isPublished): self
	{
		$this->isPublished = $isPublished;

		return $this;
	}

	public function getPosition(): int { return $this->position; }

	public function setPosition(int $position): self
	{
		$this->position = $position;

		return $this;
	}

	public function getPrice(): float { return $this->price; }

	public function getPriceWithoutVat(): float { return round(($this->price / (100 + $this->vat)) * 100, 2); }

	public function getPriceInBaseCurrency(): float { return $this->priceInBaseCurrency; }

	public function getPriceInBaseCurrencyWithoutVat(): float { return round(($this->priceInBaseCurrency / (100 + $this->vat)) * 100, 2); }

	public function setPrice(float $price): self
	{
		$this->price = $price;

		return $this;
	}

	public function setCart(Cart &$cart): self
	{
		$this->cart = &$cart;

		return $this;
	}

	/**
	 * vrati cenu podle kosiku:
	 *  - podle celkove ceny objednavky muze byt doprava zdarma
	 *  - kdyz je v kosiku produkt se stitkem "doprava zdarma"
	 *
	 * @param Cart $cart
	 * @param bool $returnInBaseCurrency
	 *
	 * @return float
	 */
	public function getPriceActual(?Cart $cart = null, bool $returnInBaseCurrency = false): float
	{
		$cart  = $cart ?: $this->cart;
		$price = $this->prepareActualPrice($cart, $returnInBaseCurrency);

		if ($cart->payment && $cart->spedition) {
			foreach ($cart->paymentSpeditionFreeCombinations as $ps) {
				if ($ps->getPayment()->getId() == $cart->payment->getId() && $ps->getSpedition()->getId() == $cart->spedition->getId())
					return 0;
			}
		}

		return $price;
	}

	public function getPriceActualWithoutVat(?Cart $cart = null, bool $returnInBaseCurrency = false): float
	{
		return round(($this->getPriceActual($cart, $returnInBaseCurrency) / (100 + $this->vat)) * 100, 2);
	}

	public function getSpeditionChoicePrice(?Cart $cart = null, bool $returnInBaseCurrency = false): float
	{
		$cart = $cart ?: $this->cart;

		return $this->prepareActualPrice($cart, $returnInBaseCurrency);
	}

	protected function prepareActualPrice(?Cart $cart = null, bool $returnInBaseCurrency = false): float
	{
		$cart     = $cart ?: $this->cart;
		$freeFrom = $this->getFreeFrom();

		if ($freeFrom !== null && $freeFrom <= $cart->getCartItemsPrice()) {
			return 0;
		}

		$freeDelivery = true;
		foreach ($cart->getCartItems() as $item) {
			if ($item->freeDelivery !== true) {
				$freeDelivery = false;
			}
		}

		if ($freeDelivery) {
			return 0;
		}

		return $returnInBaseCurrency ? $this->getPriceInBaseCurrency() : $this->getPrice();
	}

	public function getFreeFrom(): ?float { return $this->freeFrom; }

	public function getFreeFromInBaseCurrency(): float { return $this->freeFromInBaseCurrency; }

	public function setFreeFrom(?float $freeFrom): self
	{
		$this->freeFrom = $freeFrom;

		return $this;
	}

	public function getAvailableFrom(): int { return $this->availableFrom ?: 0; }

	public function setAvailableFrom(int $availableFrom): self
	{
		$this->availableFrom = $availableFrom;

		return $this;
	}

	public function getAvailableTo(): int { return $this->availableTo ?: 999999999; }

	public function setAvailableTo(int $availableTo): self
	{
		$this->availableTo = $availableTo;

		return $this;
	}

	public function getImage(): ?string { return $this->image; }

	public function setImage(string $image): self
	{
		$this->image = $image;

		return $this;
	}
}
