<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Model\Dao;

use EshopCatalog\Model\Entities\Availability;
use EshopOrders\Model\Helpers\OrderHelper;

class Cart
{

	/** @var int */
	public $id;

	/** @var string */
	public $ident;

	/** @var CartItem[] */
	public $cartItems;

	/** @var Spedition|null */
	public $spedition;

	/** @var Payment */
	public $payment;

	/** @var Discount[] */
	public $discounts = [];

	/** @var Gift[] */
	public $gifts = [];

	public ?int $orderGiftId = null;

	public float $minimalOrderPrice = 0;

	/** @var array */
	protected $cCalculatedDiscounts = [];

	/** @var PaymentSpedition[] */
	public array $paymentSpeditionFreeCombinations = [];

	public function setId(int $id)
	{
		$this->id = $id;

		return $this;
	}

	public function setIdent($ident)
	{
		$this->ident = $ident;

		return $this;
	}

	public function setCartItems($cartItems)
	{
		$this->cartItems = $cartItems;

		return $this;
	}

	public function getId(): int
	{
		return $this->id;
	}

	public function getIdent()
	{
		return $this->ident;
	}

	/** @return CartItem[] */
	public function getCartItems(): array
	{
		return $this->cartItems === null ? [] : $this->cartItems;
	}

	public function getCartItemByProductId(int $id): ?CartItem
	{
		foreach ($this->getCartItems() as $item) {
			if ($item->productId === $id) {
				return $item;
			}
		}

		return null;
	}

	public function getCartItemByIdent(string $ident): ?CartItem
	{
		foreach ($this->getCartItems() as $item) {
			if ($item->ident === $ident) {
				return $item;
			}
		}

		return null;
	}

	/** @return Gift[] */
	public function getGifts(): array { return $this->gifts; }

	public function addGift(Gift $gift): self
	{
		$this->gifts[$gift->id] = $gift;

		return $this;
	}

	public function getCartItemsPrice($withVat = true, bool $useBaseCurrency = false): float
	{
		$priceTotal = 0;
		foreach ($this->cartItems as $item) {
			$priceTotal += $withVat ? $item->getTotalPrice($useBaseCurrency) : $item->getTotalPriceWithoutVat($useBaseCurrency);
		}

		return $priceTotal;
	}

	public function getPriceTotal(): float
	{
		$priceTotal = $this->getCartItemsPrice(true);
		$priceTotal += $this->calculateDiscounts(true);

		if ($this->spedition)
			$priceTotal += $this->spedition->getPriceActual($this);
		if ($this->payment)
			$priceTotal += $this->payment->getPriceActual($this);

		return $priceTotal;
	}

	public function getPriceTotalWithoutVat(): float
	{
		$speditionPrice      = $this->spedition ? $this->spedition->getPriceActual($this) : 0;
		$paymentPrice        = $this->payment ? $this->payment->getPriceActual($this) : 0;
		$itemsPrice          = $this->getCartItemsPrice(true) + $speditionPrice + $paymentPrice;
		$sale                = $this->calculateDiscounts(true);
		$itemsPriceAfterSale = $itemsPrice + $sale;

		$dphRates = [];
		foreach ($this->getCartItems() as $item) {
			if (!isset($dphRates[$item->getVatRate()]))
				$dphRates[$item->getVatRate()] = 0;
			$dphRates[$item->getVatRate()] += $item->getTotalPrice();
		}

		foreach (['spedition', 'payment'] as $c) {
			if ($this->$c) {
				if (!isset($dphRates[$this->$c->vat]))
					$dphRates[$this->$c->vat] = 0;
				$dphRates[$this->$c->vat] += $this->$c->getPriceActual($this);
			}
		}

		return OrderHelper::calculatePriceWithoutVatWithSales($itemsPrice, $itemsPriceAfterSale, $dphRates);
	}

	public function getPriceTotalWithoutSpedition($withVat = true): float
	{
		$itemsPrice          = $this->getCartItemsPrice(true);
		$sale                = $this->calculateDiscounts(true);
		$itemsPriceAfterSale = $itemsPrice + $sale;

		if ($withVat)
			return $itemsPriceAfterSale;

		$dphRates = [];
		foreach ($this->getCartItems() as $item) {
			if (!isset($dphRates[$item->getVatRate()]))
				$dphRates[$item->getVatRate()] = 0;
			$dphRates[$item->getVatRate()] += $item->getTotalPrice();
		}

		return OrderHelper::calculatePriceWithoutVatWithSales($itemsPrice, $itemsPriceAfterSale, $dphRates);
	}

	public function getItemsCount(): int
	{
		$count = 0;
		foreach ($this->cartItems as $item) {
			$count += $item->getQuantity();
		}

		return $count;
	}

	public function addDiscount(string $key, Discount $discount): self
	{
		$this->discounts[$key] = $discount;

		return $this;
	}

	public function removeDiscount(string $key): self
	{
		unset($this->discounts[$key]);

		return $this;
	}

	public function getDiscount(string $key): ?Discount { return $this->discounts[$key] ?? null; }

	/** @return Discount[] */
	public function getDiscounts(): iterable
	{
		// Přepočítá slevu, může tam být uložena sleva bez DPH
		$this->calculateDiscounts();

		$discounts = [];
		foreach ($this->discounts as $k => $discount) {
			$discounts[$k] = $discount;
		}

		return $discounts;
	}

	public function setSpedition(Spedition $spedition): self
	{
		$spedition->setCart($this);
		$this->spedition = $spedition;

		return $this;
	}

	public function setPayment(Payment $payment): self
	{
		$payment->setCart($this);
		$this->payment = $payment;

		return $this;
	}

	public function calculateDiscounts($withVat = true): float
	{
		$k = (string) $withVat;

		if (!isset($this->cCalculatedDiscounts[$k])) {
			$price = 0;
			foreach ($this->discounts as $discount)
				$price += $discount->calculateDiscount();

			$this->cCalculatedDiscounts[$k] = $price;
		}

		return $this->cCalculatedDiscounts[$k];
	}

	public function calculateDiscount(Discount $discount, $withVat = true, bool $useBaseCurrency = false): float
	{
		return $discount->calculateDiscount($useBaseCurrency);
	}

	/** @return CartItem[] */
	public function getProductsForDiscount(): array
	{
		$items = [];

		foreach ($this->cartItems as $k => $item) {
			if ($item->discountDisabled)
				continue;

			$items[$k] = $item;
		}

		return $items;
	}

	/**
	 * @return bool
	 */
	public function hasOversizedProduct(): bool
	{
		foreach ($this->cartItems as $item) {
			if ($item->product && $item->product->isOversize === true)
				return true;
		}

		return false;
	}

	/**
	 * @return bool
	 */
	public function hasDisabledPickUpSpedition(): bool
	{
		foreach ($this->cartItems as $item) {
			if ($item->product && $item->product->disablePickUpSpedition === true)
				return true;
		}

		return false;
	}

	public function getOrderGift(): ?Gift
	{
		if (!$this->orderGiftId)
			return null;

		return $this->gifts[$this->orderGiftId] ?? null;
	}

	public function getRemainingValueToCompleteOrder(): float
	{
		return $this->minimalOrderPrice > 0 ? $this->minimalOrderPrice - $this->getCartItemsPrice() : 0;
	}

	public function hasPreorderProduct(): bool
	{
		foreach ($this->getCartItems() as $item)
			if ($item->getProduct() && $item->getProduct()->getAvailability()->getIdent() === Availability::PREORDER)
				return true;

		return false;
	}

	/**
	 * Grams
	 * @return int
	 */
	public function getTotalWeight(): int
	{
		$weight = 0;

		foreach ($this->getCartItems() as $item) {
			$weight += $item->getTotalWeight();
		}

		return $weight;
	}

	public function getTotalWeightInKG(): float
	{
		$totalWeight = $this->getTotalWeight();

		if ($totalWeight > 0) {
			return round($totalWeight / 1000, 2);
		}

		return 0;
	}
}
