<?php declare(strict_types = 1);

namespace EshopOrders\Model\Entities\Invoice;

use Doctrine\ORM\Mapping as ORM;
use Core\Model\Entities\TId;

/**
 * @ORM\Table("eshop_orders__invoice_product")
 * @ORM\Entity
 */
class Product
{
	use TId;

	/**
	 * @var string|null
	 * @ORM\Column(type="string", nullable = true)
	 */
	public $name;

	/**
	 * @var string|null
	 * @ORM\Column(name="code1", type="string", nullable=true)
	 */
	public ?string $code1 = null;

	/**
	 * Parametr podstatny jen pri vytvareni faktury, jinak nepouzivat
	 * @var integer|null
	 * @ORM\Column(type="integer", nullable=true)
	 */
	public $productId;

	/**
	 * @var integer
	 * @ORM\Column(type="integer", options={"default": 0})
	 */
	public $quantity = 0;

	/**
	 * @var float
	 * @ORM\Column(type="decimal", precision=10, scale=2)
	 */
	public $price = 0;

	/**
	 * @var int
	 * @ORM\Column(type="smallint")
	 */
	protected $vatRate;

	/**
	 * @var string
	 * @ORM\Column(type="text", nullable=true)
	 */
	public $imageBase64;

	/**
	 * @var InvoiceData
	 *
	 * @ORM\ManyToOne(targetEntity="InvoiceData")
	 * @ORM\JoinColumn(name="invoice_data_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	protected $invoiceData;

	/**
	 * @var double
	 * @ORM\Column(name="recycling_fee", type="decimal", precision=10, scale=2, nullable=true)
	 */
	public $recyclingFee;

	/**
	 * @ORM\Column(name="more_data", type="array", nullable=true)
	 */
	protected ?array $moreData = [];

	/**
	 * Product constructor.
	 *
	 * @param InvoiceData $invoiceData
	 */
	public function __construct(InvoiceData $invoiceData)
	{
		$this->invoiceData = $invoiceData;
	}

	/**
	 * @return string
	 */
	public function getImageBase64(): string
	{
		return str_replace('png', 'jpeg', $this->imageBase64); // png mi nechce mpdf zobrazit
	}

	/**
	 * @return float
	 */
	public function getPrice(): float
	{
		return (float) $this->price;
	}

	public function getQuantity(): int { return $this->quantity ?: 0; }

	/**
	 * @return float
	 */
	public function getPriceTotal(): float
	{
		return (float) $this->getPrice() * $this->quantity;
	}

	/**
	 * @return float
	 */
	public function getTotalPriceWithoutVat(): float
	{
		return round($this->getPriceWithoutVat() * $this->quantity, 2);
	}

	/**
	 * @return float
	 */
	public function getPriceWithoutVat(): float
	{
		return round($this->price / (1 + ($this->vatRate / 100)), 2);
	}

	public function getVatRate(): int { return (int) $this->vatRate; }

	public function setVatRate(int $rate) { $this->vatRate = $rate; }

	public function getRecyclingFeeWithoutVat(): float
	{
		return (float) $this->recyclingFee;
	}

	public function getTotalRecyclingFeeWithoutVat(): float
	{
		return (float) $this->recyclingFee * $this->quantity;
	}

	public function getMoreData(): array { return $this->moreData ?? []; }

	public function getMoreDataValue(string $key, $default = null)
	{
		return $this->moreData[$key] ?? $default;
	}

	public function setMoreData(array $data): self
	{
		$this->moreData = $data;

		return $this;
	}
}
