<?php declare(strict_types = 1);

namespace EshopOrders\Model\Listeners;

use Core\Model\Event\Event;
use Core\Model\Event\EventDispatcher;
use Doctrine\ORM\Mapping as ORM;
use Doctrine\ORM\Event\LifecycleEventArgs;
use EshopOrders\Model\Entities\Invoice;
use EshopOrders\Model\InvoiceConfigRepository;
use Exception;

class InvoiceListener
{
	protected InvoiceConfigRepository $invoiceConfigRepository;
	protected EventDispatcher $eventDispatcher;

	/**
	 * InvoiceListener constructor.
	 * @param InvoiceConfigRepository $invoiceConfigRepository
	 * @param EventDispatcher $eventDispatcher
	 */
	public function __construct(InvoiceConfigRepository $invoiceConfigRepository, EventDispatcher $eventDispatcher)
	{
		$this->invoiceConfigRepository = $invoiceConfigRepository;
		$this->eventDispatcher = $eventDispatcher;
	}

	/**
	 * @ORM\PrePersist
	 *
	 * @param Invoice            $invoice
	 * @param LifecycleEventArgs $event
	 *
	 * @throws Exception
	 */
	public function prePersist(Invoice $invoice, LifecycleEventArgs $event): void
	{
		$entityManager = $event->getEntityManager();

		$sellerId = $invoice->seller->getId();
		$event = new Event(['order' => $invoice->order]);
		$this->eventDispatcher->dispatch($event, self::class . '::beforeIncreaseStartNumber');

		if (isset($event->data['seller']) && $event->data['seller'] !== null) {
			$sellerId = $event->data['seller']->getId();
		}

		$config = $this->invoiceConfigRepository->getConfigBySeller($sellerId);

		$numericalSeries = $invoice->order->isCorrectiveTaxDocument ? $config->correctiveNumericalSeries : $config->numericalSeries;
		$numericalSeries->increaseStartNumber();

		$entityManager->persist($numericalSeries);
		$entityManager->flush($numericalSeries);
	}
}
