<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Order;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\Model\Entities\Product;
use Nette\Application\UI\Form;
use Nette\Application\UI\ITemplateFactory;
use Nette\Http\IResponse;
use Nette\Utils\ArrayHash;
use Nette\Mail\IMailer;
use Nette\Mail\Message;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderItem;
use EshopOrders\Model\Entities\OrderItemTexts;
use EshopOrders\Model\Orders;
use EshopOrders\Model\OrderItems;

class OrderItemForm extends BaseControl
{
	/** @var int @persistent */
	public $orderItemId;

	/** @var OrderItem */
	public $orderItem;

	/** @var OrderItemTexts */
	public $orderItemTexts;

	/** @var Order */
	public $order;

	/** @var Orders */
	protected $ordersService;

	/** @var Items */
	protected $orderItemsService;

	public function __construct(Orders $orders, OrderItems $items)
	{
		$this->ordersService     = $orders;
		$this->orderItemsService = $items;
	}

	public function render()
	{
		$this->template->item = $this->orderItem;
		$this->template->render($this->getTemplateFile());
	}

	public function attached($presenter)
	{
		parent::attached($presenter);

		if ($this->orderItemId && !$this->orderItem) {
			$this->setOrderItem($this->orderItemId);
		}
	}

	/*******************************************************************************************************************
	 * ======================== Handler
	 */

	/**
	 * @param int $itemId
	 */
	public function handleDelete($itemId)
	{
		$presenter = $this->getPresenter();
		$this->orderItemsService->remove($itemId);
		$presenter->flashMessageSuccess('default.removed');
		$presenter->redrawControl('flashes');

		$presenter['orderForm']->redrawControl('items');
		$presenter['orderForm']->redrawControl('priceTotal');
	}

	/*******************************************************************************************************************
	 * ======================== Components
	 */

	protected function createComponentForm()
	{
		$form = $this->createForm();

		$form->addInteger('quantity', 'eshopOrders.default.quantity')->setRequired()->addRule(Form::MIN, 'Počet musí být alespoň %d', 1);
		$form->addText('price', 'eshopOrders.default.pricePiece')->setRequired()->addRule(Form::FLOAT, 'Cena musí být číslo');

		$form->addSaveCancelControl();
		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
		try {
			if ($this->orderItem) {
				$orderItem    = $this->orderItem;
				$flashMessage = 'eshopOrders.itemForm.edited';
			} else {
				$orderItem    = new OrderItem($this->em->getReference(Product::class, $values->product_id), $this->order);
				$flashMessage = 'eshopOrders.itemForm.added';
			}

			$orderItem->setQuantity($values->quantity);
			$orderItem->setPrice($values->price);

			$this->em->persist($orderItem)->flush();

			$this->getPresenter()->flashMessageSuccess($flashMessage);
		} catch (\Exception $e) {
			$form->addError($e->getMessage());
			$this->redrawControl('form');

			return false;
		}
	}

	public function setOrder($orderId)
	{
		$this->order = $this->ordersService->get($orderId);
	}

	public function setOrderItem($orderItemId)
	{
		$this->orderItem      = $this->orderItemsService->get($orderItemId);
		$this->orderItemTexts = $this->orderItem->getOrderItemText('cs');
		$this->orderItemId    = $orderItemId;

		if ($this->orderItem) {
			$i = $this->orderItem;
			$this['form']->setDefaults([
				'quantity' => $i->getQuantity(),
				'price'    => $i->getPrice(),
			]);
		}
	}
}
