<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Components\Customer;

use Core\Model\Countries;
use Core\Model\UI\Form\BaseForm;
use Nette\Forms\Form;
use Nette\Http\IResponse;
use Nette\Utils\ArrayHash;
use Core\Model\UI\BaseControl;
use EshopOrders\Model\Entities\Customer;
use EshopOrders\Model\Entities\CustomerAddress;
use Users\Model\Entities\User;

class ProfileForm extends BaseControl
{
	/** @var User */
	public $user;

	/** @var Customer */
	public $customer;

	/** @var Countries */
	protected $countriesService;

	public function __construct(Countries $countries)
	{
		$this->countriesService = $countries;
	}

	public function render()
	{
		if (!$this->user)
			return '';

		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();

		$form->addText('name', 'default.name');
		$form->addText('lastname', 'default.lastName')->setRequired();
		$form->addPassword('password', 'default.password');
		$form->addEmail('email', 'default.email')->setRequired();
		$form->addText('phone', 'default.phone');

		$countries = $this->countriesService->getAllNameColumn();
		$form->addSelect('country', 'eshopOrders.orderPage.country', $countries);
		$form->addText('street', 'default.street');
		$form->addText('city', 'default.city');
		$form->addText('postal', 'default.zipNumber');
		$form->addText('company', 'eshopOrders.orderPage.company');
		$form->addText('idNumber', 'eshopOrders.orderPage.idNumber');
		$form->addText('vatNumber', 'eshopOrders.orderPage.vatNumber');

		$form->addSubmit('submit', 'default.save');
		$form->addCancel('cancel', 'default.cancel');

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(Form $form, ArrayHash $values)
	{
		if ($form->isSubmitted()->getName() === 'cancel') {
			$this->getPresenter()->redirect(':EshopOrders:Front:Customer:');
		}

		try {
			$user     = $this->user;
			$customer = $this->customer;
			$address  = $customer->getAddressDelivery();

			$user->setName($values->name);
			$user->setLastname($values->lastname);
			$user->setEmail($values->email);

			$customer->setPhone($values->phone);

			if (!$address) {
				$address = new CustomerAddress($customer);
			}

			$country = $values->country ? $this->countriesService->getReference($values->country) : null;
			if ($country)
				$address->setCountry($country);
			$address->setStreet($values->street);
			$address->setCity($values->city);
			$address->setPostal($values->postal);
			$address->setCompany($values->company);
			$address->setIdNumber($values->idNumber);
			$address->setVatNumber($values->vatNumber);

			$customer->setAddressDelivery($address);

			if ($values->password != '')
				$user->setPassword($values->password);

			$this->em->persist($user);
			$this->em->persist($customer);
			$this->em->persist($address);
			$this->em->flush();
			$this->getPresenter()->flashMessageSuccess('users.profileForm.profileUpdated');
		} catch (\Exception $e) {
			$form->addError($e->getMessage());

			return false;
		}

		return true;
	}

	public function setUser($user, $customer)
	{
		$this->user     = $user;
		$this->customer = $customer;

		if ($this->user && $this->customer) {
			$defaults = [
				'name'     => $this->user->getName(),
				'lastname' => $this->user->getLastname(),
				'email'    => $this->user->getEmail(),
				'phone'    => $this->customer->getPhone(),
			];
			if ($address = $this->customer->getAddressDelivery()) {
				$defaults['street']    = $address->getStreet();
				$defaults['city']      = $address->getCity();
				$defaults['postal']    = $address->getPostal();
				$defaults['company']   = $address->getCompany();
				$defaults['idNumber']  = $address->getIdNumber();
				$defaults['vatNumber'] = $address->getVatNumber();

				if ($address->getCountry() && array_key_exists($address->getCountry()->getId(), $this['form']['country']->getItems()))
					$defaults['country'] = $address->getCountry()->getId();
			}

			$this['form']->setDefaults($defaults);

			$this['form']['password']->setRequired(false);
		} else {
			$this->getPresenter()->error(null, IResponse::S404_NOT_FOUND);
		}
	}
}

