<?php declare(strict_types=1);

namespace EshopOrders\AdminModule\Components\Invoice;

use Core\Model\Pdf\Pdf;
use Core\Model\Templating\Filters\Price;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use EshopOrders\Model\Entities\Invoice;
use EshopOrders\Model\Invoices;
use Nette\Utils\Html;
use Ublaboo\DataGrid\Exception\DataGridException;

class InvoicesGrid extends BaseControl
{
	/** @var Invoices */
	protected $invoices;

	/** @var Price */
	protected $priceFilter;

	/**
	 * InvoiceConfigForm constructor.
	 * @param Invoices $invoices
	 * @param Price $priceFilter
	 */
	public function __construct(Invoices $invoices, Price $priceFilter)
	{
		parent::__construct();
		$this->invoices = $invoices;
		$this->priceFilter = $priceFilter;
	}

	/**
	 * @return BaseDataGrid
	 * @throws DataGridException
	 */
	public function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$grid->setDataSource($this->invoices->getQueryBuilder());

		// columns
		$grid->addColumnText('ident', 'eshopOrders.invoicesGrid.ident');
		$grid->addColumnText('subscriber', 'eshopOrders.invoicesGrid.subscriber')->setRenderer(static function (Invoice $invoice) {
			$address = $invoice->order->getAddressInvoice();

			if ($address === null) {
				return '';
			}

			return $address->getCompany() ?? sprintf('%s %s', $address->getFirstName(), $address->getLastName());
		});
		$grid->addColumnDateTime('fromDate', 'eshopOrders.invoicesGrid.fromDate')->setRenderer(static function (Invoice $invoice) {
			return $invoice->order->getCreatedTime()->format('d. m. Y');
		});
		$grid->addColumnDateTime('dueDate', 'eshopOrders.invoicesGrid.dueDate');
		$grid->addColumnText('total', 'eshopOrders.invoicesGrid.totalPrice')->setRenderer(function (Invoice $invoice) {
			return $this->priceFilter->format($invoice->order->getPrice());
		});

		// actions
		$grid->addAction('pdf', '')->setRenderer(function (Invoice $invoice) {
			return Html::el('a', ['href' => $this->presenter->link('pdf!', $invoice->getId()), 'class' => 'btn btn-danger btn-xs', 'target' => '_blank'])
					   ->addHtml(Html::el('i', ['class' => 'fas fa-file-pdf']))
					   ->addHtml('&nbsp;')
					   ->addText($this->t('eshopOrders.invoicesGrid.pdf'));
		});

		return $grid;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

}