<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Presenters;

use EshopOrders\AdminModule\Components\Order\IOrderItemAddFormFactory;
use EshopOrders\AdminModule\Model\Event\OrderNavigationCallbackEvent;
use EshopOrders\AdminModule\Model\Event\OrdersNavigationEvent;
use Nette\Http\IResponse;
use EshopOrders\AdminModule\Components\Order\IOrderFormFactory;
use EshopOrders\AdminModule\Components\Order\IOrdersGridFactory;
use EshopOrders\AdminModule\Components\Order\IOrderStatusFormFactory;
use EshopOrders\AdminModule\Components\Order\IOrderItemFormFactory;
use EshopOrders\AdminModule\Components\Order\IOrderSpeditionFormFactory;
use EshopOrders\AdminModule\Components\Order\IOrderPaymentFormFactory;
use EshopOrders\AdminModule\Components\Order\IOrderAddressFormFactory;
use EshopOrders\Model\Entities\Order;

class DefaultPresenter extends BasePresenter
{
	protected function startup()
	{
		parent::startup();
		$this['title']->setTitle($this->translator->translate('eshopOrders.title.orders'));
		$this->setHeader('eshopOrders.title.orders', 'fa fa-shopping-cart');
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	/** Vyrenderuje vyskakovaci formular pro pridani stavu k objednavce
	 *
	 * @param int $orderId ID objednavky
	 */
	public function handleOrderStatusForm()
	{
		$this->template->modal = 'orderStatusForm';
		$this->redrawControl('modal');

		$control = $this['orderStatusForm'];
		$this->setTitle($this->t('eshopOrders.title.addOrderStatus'));
		$this->template->modalTitle = $this['title']->getTitle();
	}

	/** Vyrenderuje vyskakovaci formular pro editaci položky (cena a pocet ks)
	 *
	 * @param int $itemId ID polozky
	 */
	public function handleOrderItemForm($itemId = null)
	{
		$this->template->modal = 'orderItemForm';
		$this->redrawControl('modal');

		$control = $this['orderItemForm'];
		if ($itemId) {
			$control->setOrderItem($itemId);
			$this->setTitle($this->t('eshopOrders.title.editOrderItem'), $itemId);
		} else {
			$this->setTitle($this->t('eshopOrders.title.addOrderItem'));
		}
		$this->template->modalTitle = $this['title']->getTitle();
	}

	/** Vyrenderuje vyskakovaci formular pro pridani stavu k objednavce
	 *
	 * @param int $itemId ID objednavky
	 */
	public function handleOrderItemAddForm()
	{
		$this->template->modal = 'orderItemAddForm';
		$this->redrawControl('modal');

		$this->setTitle($this->t('eshopOrders.title.addOrderItem'), ' ');
		$this->template->modalTitle = $this['title']->getTitle();
	}

	/** Vyrenderuje vyskakovaci formular pro editaci dopravy v objednavce
	 *
	 * @param int $itemId ID objednavky
	 */
	public function handleOrderSpeditionForm()
	{
		$this->template->modal = 'orderSpeditionForm';
		$this->redrawControl('modal');

		$control = $this['orderSpeditionForm'];

		$this->setTitle($this->t('eshopOrders.title.editOrderSpedition'));

		$this->template->modalTitle = $this['title']->getTitle();
	}

	/** Vyrenderuje vyskakovaci formular pro editaci platby v objednavce
	 *
	 * @param int $itemId ID objednavky
	 */
	public function handleOrderPaymentForm()
	{
		$this->template->modal = 'orderPaymentForm';
		$this->redrawControl('modal');

		$control = $this['orderPaymentForm'];

		$this->setTitle($this->t('eshopOrders.title.editOrderPayment'));

		$this->template->modalTitle = $this['title']->getTitle();
	}

	/** Vyrenderuje vyskakovaci formular pro editaci dorucovaci adresy
	 */
	public function handleAddressDeliveryForm()
	{
		$this->template->modal = 'orderAddressDeliveryForm';
		$this->redrawControl('modal');

		$this->setTitle($this->t('eshopOrders.title.editAddressDelivery'));

		$this->template->modalTitle = $this['title']->getTitle();
	}

	/** Vyrenderuje vyskakovaci formular pro editaci dorucovaci adresy
	 */
	public function handleAddressInvoiceForm()
	{
		$this->template->modal = 'orderAddressInvoiceForm';
		$this->redrawControl('modal');

		$this->setTitle($this->t('eshopOrders.title.editAddressInvoice'));

		$this->template->modalTitle = $this['title']->getTitle();
	}

	public function handleNavigationCallback(string $key)
	{
		$this->eventDispatcher->dispatch('eshopOrders.admin.ordersDefaultNavigationCallback',
			new OrderNavigationCallbackEvent($key, $this));
	}

	/*******************************************************************************************************************
	 * ============================== Actions
	 */

	public function actionDefault()
	{
		$navigation = [];

		$this->eventDispatcher->dispatch('eshopOrders.admin.ordersDefaultNavigation', new OrdersNavigationEvent($navigation));
		foreach ($navigation as $k => $nav) {
			if (is_array($nav['link']))
				$navigation[$k]['link'] = $this->link($nav['link'][0], $nav['link'][1]);
		}

		// invoices navigation item
		$navigation[] = [
			'title' => 'eshopOrders.menu.invoices',
			'link'  => $this->link('Invoice:default'),
			'ico'   => 'fas fa-file-invoice-dollar'
		];

		// invoice settings navigation item
		$navigation[] = [
			'title' => 'eshopOrders.menu.invoiceSettings',
			'link'  => $this->link('Invoice:invoiceSettings'),
			'ico'   => 'fas fa-cogs'
		];

		$this['navigation']->setData(['header' => $navigation]);
	}

	public function actionEditOrder($id)
	{
		$this['navigation']->setData(['header' => [
			[
				'title' => 'eshopOrders.menu.editAddressDelivery',
				'link'  => $this->link('addressDeliveryForm!'),
				'ico'   => 'plus',
				'class' => 'ajax',
			],
			[
				'title' => 'eshopOrders.menu.editAddressInvoice',
				'link'  => $this->link('addressInvoiceForm!', ['type' => 'invoice']),
				'ico'   => 'plus',
				'class' => 'ajax',
			],
			[
				'title' => 'eshopOrders.menu.addItem',
				'link'  => $this->link('orderItemAddForm!'),
				'ico'   => 'plus',
				'class' => 'ajax',
			],
			[
				'title' => 'eshopOrders.menu.addStatus',
				'link'  => $this->link('orderStatusForm!'),
				'ico'   => 'plus',
				'class' => 'ajax',
			],
		]]);

		$order = $this->em->getRepository(Order::class)->find($id);

		if (!$order)
			$this->error(null, IResponse::S404_NOT_FOUND);

		$this->setTitle(
			$this->translator->translate('eshopOrders.title.editOrder'),
			"(" . $order->addressDelivery->firstName . " " . $order->addressDelivery->lastName . ")"
		);
		$this->setHeader('eshopOrders.title.editOrder', 'fa fa-shopping-cart');
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */

	protected function createComponentOrdersGrid(IOrdersGridFactory $factory)
	{
		return $factory->create();
	}

	protected function createComponentOrderForm(IOrderFormFactory $factory)
	{
		$control = $factory->create();

		if ($this->getParameter('id'))
			$control->setOrder($this->getParameter('id'));

		return $control;
	}

	protected function createComponentOrderStatusForm(IOrderStatusFormFactory $factory)
	{
		$control = $factory->create();
		$control['form']->setAjax();

		if ($this->getParameter('id'))
			$control->setOrder($this->getParameter('id'));

		$control['form']->onSuccessSave[]         = function() {
			$this->redrawControl('flashes');
			$this['orderForm']->redrawControl('statuses');
		};
		$control['form']->onSuccessSaveAndClose[] = function() {
			$this->payload->hideModal = true;
			$this->redrawControl('flashes');
			$this['orderForm']->redrawControl('statuses');
		};

		$control['form']['saveControl']->closeModalOnCancel();

		return $control;
	}

	protected function createComponentOrderItemForm(IOrderItemFormFactory $factory)
	{
		$control = $factory->create();
		$control['form']->setAjax();

		if ($this->getParameter('id'))
			$control->setOrder($this->getParameter('id'));

		$control['form']->onSuccessSave[]         = function() {
			$this->redrawControl('flashes');
			$this['orderForm']->redrawControl('items');
			$this['orderForm']->redrawControl('priceTotal');
		};
		$control['form']->onSuccessSaveAndClose[] = function() {
			$this->payload->hideModal = true;
			$this->redrawControl('flashes');
			$this['orderForm']->redrawControl('items');
			$this['orderForm']->redrawControl('priceTotal');
		};

		$control['form']['saveControl']->closeModalOnCancel();

		return $control;
	}

	protected function createComponentOrderItemAddForm(IOrderItemAddFormFactory $factory)
	{
		$control = $factory->create();
		$control['form']->setAjax();

		if ($this->getParameter('id'))
			$control->setOrder($this->getParameter('id'));

		$control['form']->onSuccessSave[]         = function() {
			$this->redrawControl('flashes');
			$this['orderForm']->redrawControl('items');
			$this['orderForm']->redrawControl('priceTotal');
		};
		$control['form']->onSuccessSaveAndClose[] = function() {
			$this->payload->hideModal = true;
			$this->redrawControl('flashes');
			$this['orderForm']->redrawControl('items');
			$this['orderForm']->redrawControl('priceTotal');
		};

		$control['form']['saveControl']->closeModalOnCancel();

		return $control;
	}

	protected function createComponentOrderSpeditionForm(IOrderSpeditionFormFactory $factory)
	{
		$control = $factory->create();
		$control['form']->setAjax();

		if ($this->getParameter('id'))
			$control->setOrder($this->getParameter('id'));

		$control['form']->onSuccessSave[]         = function() {
			$this->redrawControl('flashes');
			$this['orderForm']->redrawControl('spedition');
			$this['orderForm']->redrawControl('priceTotal');
		};
		$control['form']->onSuccessSaveAndClose[] = function() {
			$this->payload->hideModal = true;
			$this->redrawControl('flashes');
			$this['orderForm']->redrawControl('spedition');
			$this['orderForm']->redrawControl('priceTotal');
		};

		$control['form']['saveControl']->closeModalOnCancel();

		return $control;
	}

	protected function createComponentOrderPaymentForm(IOrderPaymentFormFactory $factory)
	{
		$control = $factory->create();
		$control['form']->setAjax();

		if ($this->getParameter('id'))
			$control->setOrder($this->getParameter('id'));

		$control['form']->onSuccessSave[]         = function() {
			$this->redrawControl('flashes');
			$this['orderForm']->redrawControl('payment');
			$this['orderForm']->redrawControl('priceTotal');
		};
		$control['form']->onSuccessSaveAndClose[] = function() {
			$this->payload->hideModal = true;
			$this->redrawControl('flashes');
			$this['orderForm']->redrawControl('payment');
			$this['orderForm']->redrawControl('priceTotal');
		};

		$control['form']['saveControl']->closeModalOnCancel();

		return $control;
	}

	protected function createComponentOrderAddressDeliveryForm(IOrderAddressFormFactory $factory)
	{
		$control = $factory->create();
		$control['form']->setAjax();

		if ($this->getParameter('id'))
			$control->setOrderAddress($this->getParameter('id'), 'delivery');

		$control['form']->onSuccessSave[]         = function() {
			$this->redrawControl('flashes');
			$this['orderForm']->redrawControl('addressDelivery');
			$this['orderForm']->redrawControl('addressInvoice');
		};
		$control['form']->onSuccessSaveAndClose[] = function() {
			$this->payload->hideModal = true;
			$this->redrawControl('flashes');
			$this['orderForm']->redrawControl('addressDelivery');
			$this['orderForm']->redrawControl('addressInvoice');
		};

		$control['form']['saveControl']->closeModalOnCancel();

		return $control;
	}

	protected function createComponentOrderAddressInvoiceForm(IOrderAddressFormFactory $factory)
	{
		$control = $factory->create();
		$control['form']->setAjax();

		if ($this->getParameter('id'))
			$control->setOrderAddress($this->getParameter('id'), 'invoice');

		$control['form']->onSuccessSave[]         = function() {
			$this->redrawControl('flashes');
			$this['orderForm']->redrawControl('addressDelivery');
			$this['orderForm']->redrawControl('addressInvoice');
		};
		$control['form']->onSuccessSaveAndClose[] = function() {
			$this->payload->hideModal = true;
			$this->redrawControl('flashes');
			$this['orderForm']->redrawControl('addressDelivery');
			$this['orderForm']->redrawControl('addressInvoice');
		};

		$control['form']['saveControl']->closeModalOnCancel();

		return $control;
	}

}
