<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Components\Cart;

use Core\Model\Event\ComponentTemplateEvent;
use EshopOrders\FrontModule\Model\CartFacade;
use Core\Model\UI\BaseControl;
use Nette\Application\UI\Multiplier;
use Nette\DI\Container;

class CartDetail extends BaseControl
{
	/** @var CartFacade */
	protected $cartFacade;

	/** @var ICartDetailItemFactory */
	protected $cartDetailItemFactory;

	/** @var Container */
	protected $container;

	/**
	 * CartDetail constructor.
	 * @param CartFacade $cartFacade
	 * @param ICartDetailItemFactory $cartDetailItemFactory
	 * @param Container $container
	 */
	public function __construct(CartFacade $cartFacade, ICartDetailItemFactory $cartDetailItemFactory, Container $container)
	{
		parent::__construct();
		$this->cartFacade = $cartFacade;
		$this->cartDetailItemFactory = $cartDetailItemFactory;
		$this->container = $container;

		// add discount form control to this component
		$discountsGridControl = $container->getService('eshopSales.front.discountsGridFactory');
		if ($discountsGridControl !== null) {
			$this->addComponent($discountsGridControl->create(), 'discountsGrid');
		}

		$discountFormControl = $container->getService('eshopSales.front.discountFormFactory');
		if ($discountFormControl !== null) {
			$this->addComponent($discountFormControl->create(), 'discountForm');
		}
	}

	protected function attached($presenter)
	{
		parent::attached($presenter);

		$this->eventDispatcher->addListener('eshopOrders.cartUpdateItem', function() {
			$cart = $this->cartFacade->getCart();

			if ($cart->getItemsCount())
				$this->redrawControl('wrap');
			else
				$this->getPresenter()->redrawControl('cart');
		});

		$this->eventDispatcher->addListener('eshopSales.discountCodesChanged', function () {
			$this->getParent()->redrawControl('cart');
		});
	}

	public function render(): void
	{
		$cart                 = $this->cartFacade->getCart();
		$this->template->cart = $cart;

		$freeSped                               = $this->cartFacade->countFreeSpedition();
		$this->template->freeSpeditionPrice     = $freeSped['minPrice'];
		$this->template->freeSpeditionRemaining = $freeSped['remaining'];

		$this->eventDispatcher->dispatch('eshopOrders.cartDetail.render', new ComponentTemplateEvent($this->template, $this));

		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ======================== Components
	 */

	protected function createComponentCartDetailItem(): Multiplier
	{
		return new Multiplier(function($item) {
			$control = $this->cartDetailItemFactory->create();
			$control->setItem((int) $item);

			return $control;
		});
	}

}

