<?php declare(strict_types=1);

namespace EshopOrders\Model;

use Core\Model\Helpers\BaseEntityService;
use EshopOrders\Model\Entities\Invoice;
use EshopOrders\Model\Entities\InvoiceConfig;
use EshopOrders\Model\Entities\NumericalSeries;
use Exception;
use Nette\Utils\DateTime;
use Nette\Utils\Strings;

class InvoiceConfigRepository extends BaseEntityService
{
	/** @var string */
	protected $entityClass = InvoiceConfig::class;

	/**
	 * @return InvoiceConfig
	 * @throws Exception
	 */
	public function getDefaultConfig(): InvoiceConfig
	{
		$defaultConfig = $this->getEr()->findOneBy(['ident' => InvoiceConfig::DEFAULT_CONFIG_IDENT]);

		if ($defaultConfig === null) {
			$prefix = (string) EshopOrdersConfig::load('invoice.numericalSeries.prefix');
			$digitsCount = (int) EshopOrdersConfig::load('invoice.numericalSeries.digitsCount');
			$startNumber = (int) EshopOrdersConfig::load('invoice.numericalSeries.startNumber');
			$maturity = (int) EshopOrdersConfig::load('invoice.maturity');

			$numericalSeries = new NumericalSeries($prefix, $digitsCount, $startNumber);
			$defaultConfig = new InvoiceConfig(InvoiceConfig::DEFAULT_CONFIG_IDENT, $maturity, $numericalSeries);
			$defaultConfig->numericalSeries = $numericalSeries;
			$this->save($defaultConfig);
		}

		return $defaultConfig;
	}

	/**
	 * @param InvoiceConfig $config
	 * @throws Exception
	 */
	public function save(InvoiceConfig $config): void
	{
		$this->em->persist($config);
		$this->em->flush();
	}

	/**
	 * @return DateTime
	 * @throws Exception
	 */
	public function getDueDate(): DateTime
	{
		$now = new DateTime;
		$defaultConfig = $this->getDefaultConfig();

		return $now->modify(sprintf('+ %s days', $defaultConfig->maturity));
	}

	/**
	 * @return string
	 * @throws Exception
	 */
	public function generateIdent(): string
	{
		$numericalSeries = $this->getDefaultConfig()->numericalSeries;
		$containsPrefixYearWildcards = $numericalSeries->containsPrefixYearWildcards();
		$containsPrefixMonthWildcards = $numericalSeries->containsPrefixMonthWildcards();
		$startNumber = $numericalSeries->startNumber;

		if ($containsPrefixMonthWildcards || $containsPrefixYearWildcards) {

			/** @var Invoice $lastInvoice */
			$lastInvoice = $this->em->getRepository(Invoice::class)->findOneBy([], ['ident' => 'desc']);

			if ($lastInvoice !== null) {
				$created = $lastInvoice->order->getCreatedTime();
				$now = new DateTime;

				// prefix contain month/year wildcards and current month/year is bigger then on the invoice
				if (($containsPrefixMonthWildcards && (((int) $now->format('n')) > ((int) $created->format('n')))) ||
					($containsPrefixYearWildcards && (((int) $now->format('Y')) > ((int) $created->format('Y')))))
				{
					$this->resetInvoiceCounter();
					$startNumber = 1;
				}
			}

		}

		$prefix = $numericalSeries->getRealPrefix();
		$order = Strings::padLeft($startNumber, $numericalSeries->digitsCount, '0');

		return sprintf('%s%s', $prefix, $order);
	}

	/**
	 * @throws Exception
	 */
	private function resetInvoiceCounter(): void
	{
		$numericalSeries = $this->getDefaultConfig()->numericalSeries;
		$numericalSeries->startNumber = 1;

		$this->em->persist($numericalSeries);
		$this->em->flush();
	}
}