<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Model;

use Core\Model\Helpers\BaseEntityService;
use EshopOrders\Model\Entities\CustomerAddress;
use Users\Model\Entities\User;
use Users\Model\Entities\Role;
use Users\Model\Users;
use Users\Model\Roles;
use EshopOrders\Model\Entities\Customer;
use EshopOrders\Model\Entities\Order;

/**
 * class Customers
 * @package EshopOrders\FrontModule\Model
 * 
 * @method Customer|object|null getReference($id)
 * @method Customer[]|null getAll()
 * @method Customer|null get($id)
 */
class Customers extends BaseEntityService
{
	const CUSTOMER = 'customer';
	
	protected $entityClass = Customer::class;
	
	/** @var Roles */
	protected $rolesService;
	
	/** @var Users */
	private $usersService;
	
	public function __construct(Users $users, Roles $roles)
	{
		$this->usersService = $users;
		$this->rolesService = $roles;
	}
	
	/**
	 * @param User $user
	 * @return Customer|null
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function getByUser(User $user)
	{
		$query = $this->getEr()->createQueryBuilder('c', 'c.id')
			->andWhere('c.user = :user')->setParameter('user', $user);
		
		$customer = $query->getQuery()->getOneOrNullResult();
		return $customer;
	}
	
	/**
	 * @param int $userId
	 * @return Customer|null
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function getByUserId($userId)
	{
		$user = $this->usersService->getReference($userId);
		return $this->getByUser($user);
	}
	
	public function getOrCreateCustomer($userId, $email, $firstName = '', $lastName = '', $phone = '')
	{
		//user
		$user = $this->usersService->getOrCreateUser($userId, $email, $firstName, $lastName);
		
		//ma user prava "customer"? Pokud ne, tak pridat prava
		if(!$user->isInRoleByIdent(self::CUSTOMER)) {
			$role = $this->rolesService->getByIdent(self::CUSTOMER);
			if($role) {
				$user->addRole($role);
				$this->em->persist($user)->flush();
			}
		}
		
		//customer
		$customer = $this->getByUser($user);
		if(!$customer) {
			$customer = new Customer($user, $phone);
			$addressDelivery = new CustomerAddress($customer);
			$customer->setAddressDelivery($addressDelivery);
			$this->em->persist($customer);
			$this->em->persist($addressDelivery);
			$this->em->flush();
		}
		
		return $customer;
	}
}
