<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Model;

use Core\Model\Helpers\BaseEntityService;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\Entities\Status;
use Users\Model\Entities\User;
use Users\Model\Http\UserStorage;
use EshopOrders\Model\Statuses as PublicStatuses;

/**
 * @method Status|object|null getReference($id)
 * @method Status[]|null getAll()
 * @method Status|null get($id)
 */
class Statuses extends BaseEntityService
{
	protected $entityClass = Status::class;

	protected $cForSelectOption;

	/** @var UserStorage */
	protected $userStorage;

	/** @var PublicStatuses */
	protected $publicStatuses;

	public function __construct(UserStorage $userStorage, PublicStatuses $statuses)
	{
		$this->userStorage    = $userStorage;
		$this->publicStatuses = $statuses;
	}

	public function changeStatus(array $ids, string $status): bool
	{
		$status = $this->getReference($status);

		if (!$status)
			return false;

		$this->em->beginTransaction();
		try {
			$statuses = [];
			$orders   = [];
			foreach ($this->em->getRepository(Order::class)->createQueryBuilder('o')
				         ->where('o.id IN (:ids)')->setParameter('ids', $ids)
				         ->getQuery()->getResult() as $row)
				$orders[$row->getId()] = $row;

			foreach ($ids as $id) {
				if (!$orders[$id])
					continue;

				$entity = new OrderStatus(
					$orders[$id],
					$status,
					$this->em->getReference(User::class, $this->userStorage->getIdentity()->getId()),
				);
				$this->em->persist($entity);
				$statuses[] = $entity;
			}
			$this->em->flush();
			$this->em->commit();

			foreach ($statuses as $v) {
				$this->sendOrderStatusEmail($v->getOrder(), $v);
			}
		} catch (\Exception $e) {
			if ($this->em->getConnection()->isTransactionActive())
				$this->em->rollback();

			return false;
		}

		return true;
	}

	public function getForSelectOption(): array
	{
		if ($this->cForSelectOption === null)
			$this->cForSelectOption = [null => ''] + ($this->getEr()->findPairs([], 'name') ?: []);

		return $this->cForSelectOption;
	}

	public function sendOrderStatusEmail(Order $order, OrderStatus $orderStatus): void
	{
		$this->publicStatuses->sendOrderStatusEmail($order, $orderStatus);
	}
}
