<?php declare(strict_types = 1);

namespace EshopOrders\Model;

use Core\Model\Helpers\BaseEntityService;
use Core\Model\Mailing\TemplateFactory;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\Entities\Status;
use Nette\Localization\ITranslator;
use Nette\Mail\IMailer;
use Nette\Mail\Message;

/**
 * class Statuses
 * @package EshopOrders\Model
 *
 * @method Status|object|null getReference($id)
 * @method Status|null get($id)
 */
class Statuses extends BaseEntityService
{
	protected $entityClass = Status::class;

	/** @var ITranslator */
	protected $translator;

	/** @var IMailer */
	protected $mailer;

	/** @var TemplateFactory */
	protected $mailTemplateFactory;

	public function __construct(ITranslator $translator, TemplateFactory $mailTemplateFactory, IMailer $mailer)
	{
		$this->translator          = $translator;
		$this->mailer              = $mailer;
		$this->mailTemplateFactory = $mailTemplateFactory;
	}

	public function setPosition($id, $position)
	{
		if ($item = $this->get($id)) {
			$item->setPosition($position);
			$this->em->persist($item);
			$this->em->flush();

			return true;
		}

		return false;
	}

	public function getAll()
	{
		return $this->getEr()->findBy(array(), array('position' => 'ASC'));
	}


	public function sendOrderStatusEmail(Order $order, OrderStatus $orderStatus): void
	{
		$file       = TEMPLATES_DIR . '/Front/default/EshopOrders/_emails/status.latte';
		$originFile = __DIR__ . '/../../FrontModule/_emails/status.latte';
		$template   = $this->mailTemplateFactory->create();

		if (!file_exists($file))
			$file = $originFile;

		$subject = $this->translator->translate('eshopOrders.emails.subjectStatus',
			['orderId' => $order->getId(), 'siteName' => $template->siteName]);

		$template->setFile($file);
		$template->setParameters([
			'subject'        => $subject,
			'order'          => $order,
			'orderStatus'    => $orderStatus,
			'orderId'        => $order->getId(),
			'originTemplate' => $originFile,
		]);

		$conf    = EshopOrdersConfig::load('orderForm');
		$message = new Message();
		$message->setFrom($conf['fromEmail'] ?? null, $conf['fromName'] ?? null);
		$message->addTo($conf['toEmail'] ?? null, $conf['toName'] ?? null);

		if (isset($conf['bcc']) && $conf['bcc']) {
			foreach (explode(',', $conf['bcc']) as $v) {
				$message->addBcc(trim($v));
			}
		}

		$message->setHtmlBody($template->renderToString());
		$message->addTo($order->getAddressInvoice()->getEmail(), $order->getAddressInvoice()->getFirstName() . ' ' . $order->getAddressInvoice()->getLastName());

		$this->mailer->send($message);
	}
}
