<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Customer;

use Core\Model\Countries;
use Core\Model\Helpers\EntityHelpers;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopOrders\AdminModule\Model\Payments;
use EshopOrders\AdminModule\Model\Speditions;
use EshopOrders\Components\AddressForm;
use EshopOrders\Components\IAddressFormFactory;
use EshopOrders\Model\Entities\CustomerAddress;
use EshopOrders\Model\Entities\GroupCustomers;
use EshopOrders\Model\EshopOrdersConfig;
use Nette\Http\IResponse;
use Nette\Utils\ArrayHash;
use EshopOrders\Model\Entities\Customer;
use EshopOrders\AdminModule\Model\Customers;
use Users\Model\Users;
use Users\Model\UsersConfig;

class CustomerForm extends BaseControl
{
	public ?Customer $customer = null;

	protected Customers $customersService;

	protected Countries $countries;

	protected Payments $payments;

	protected Speditions $speditions;

	protected Users $users;

	protected IAddressFormFactory $addressFormFactory;

	protected ?array $cOtherUsersForLogin = null;

	public function __construct(Customers $customers, Countries $countries, Payments $payments, Speditions $speditions,
	                            Users     $users, IAddressFormFactory $addressFormFactory)
	{
		$this->customersService   = $customers;
		$this->countries          = $countries;
		$this->payments           = $payments;
		$this->speditions         = $speditions;
		$this->users              = $users;
		$this->addressFormFactory = $addressFormFactory;
	}

	public function render(): void
	{
		$users   = [];
		$allUser = $this->users->getAllNonAdminUsersBasicData();
		foreach ($this->getOtherUsersForLogin() as $id) {
			if (isset($allUser[$id]))
				$users[$id] = $allUser[$id];
		}

		if (EshopOrdersConfig::load('allowAddressManagement') && $this->customer) {
			$this->template->addresses    = $this->customer->getAddressesAvailable()->toArray();
			$this->template->baseDelivery = $this->customer->getAddressDelivery() ? $this->customer->getAddressDelivery()->getId() : null;
			$this->template->baseInvoice  = $this->customer->getAddressInvoice() ? $this->customer->getAddressInvoice()->getId() : null;
		}

		$this->template->otherUsersForLogin = $users;
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleAddAddress(): void
	{
		$presenter = $this->getPresenter();

		$this['addressForm']->customerId = $this->customer ? $this->customer->getId() : null;
		$this->template->modalTitle      = $this->t('eshopOrders.customerAddress.addForm');
		$this->template->modal           = 'addressForm';
		$this->redrawControl('modal');
	}

	public function handleEditAddress(int $id): void
	{
		$presenter = $this->getPresenter();

		$this['addressForm']->addressId  = $id;
		$this['addressForm']->customerId = $this->customer ? $this->customer->getId() : null;
		$this->template->modalTitle      = $this->t('eshopOrders.customerAddress.editForm');
		$this->template->modal           = 'addressForm';
		$this->redrawControl('modal');
	}

	public function handleRemoveAddress(int $id): void
	{
		$address = $this->em->getRepository(CustomerAddress::class)->getReference($id);
		$this->em->remove($address);
		$this->em->flush();
		$this->getPresenter()->flashMessageSuccess('eshopOrders.customerAddress.removed');
		$this->getPresenter()->redrawControl('flashes');
		$this->redrawControl('addresses');
	}

	public function handleUpdateAddress(string $type, int $id)
	{
		$address = $this->em->getReference(CustomerAddress::class, $id);
		if ($type === 'delivery')
			$this->customer->setAddressDelivery($address);
		else if ($type === 'invoice')
			$this->customer->setAddressInvoice($address);
		$this->em->persist($this->customer);
		$this->em->flush();
		$this->getPresenter()->flashMessageSuccess('eshopOrders.customerAddress.updated');
		$this->getPresenter()->redrawControl('flashes');
		$this->redrawControl('addresses');
	}

	public function handleLoadUsers(): void
	{
		$this->getPresenter()->sendJson($this->users->getAllNonAdminUsersBasicData());
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();

		$groups = ['' => ''];
		foreach ($this->em->getRepository(GroupCustomers::class)->findPairs([], 'name', ['name'], 'id') as $k => $v)
			$groups[$k] = $v;

		$countries = $this->countries->getAllNameColumn();

		$form->addText('name', 'default.firstName', 'user.name')->setRequired()->setMaxLength(255);
		$form->addText('lastname', 'default.lastName', 'user.lastname')->setRequired()->setMaxLength(255);
		$form->addText('email', 'default.email')->setRequired()->setMaxLength(255);
		$form->addText('phone', 'default.phone')->setMaxLength(255);
		$form->addSelect('group', 'eshopOrders.customerForm.group', $groups);

		$form->addCheckboxList('disabledPayments', 'eshopOrders.customer.disabledPayments', $this->payments->getForSelectOption());
		$form->addCheckboxList('disabledSpeditions', 'eshopOrders.customer.disabledSpeditions', $this->speditions->getForSelectOption());

		if (!EshopOrdersConfig::load('allowAddressManagement')) {
			$addressDelivery = $form->addContainer('addressDelivery');
			$addressInvoice  = $form->addContainer('addressInvoice');

			$fields = ['firstName', 'lastName', 'email', 'phone', 'company', 'street', 'city', 'postal', 'idNumber',
				'vatNumber'];
			if (EshopOrdersConfig::load('customer.showIdVatNumber', false))
				$fields[] = 'idVatNumber';
			$fields[] = 'country';

			foreach ($fields as $v) {
				foreach (['addressDelivery', 'addressInvoice'] as $com) {
					if ($v == 'email')
						${$com}->addEmail($v, 'eshopOrders.default.' . $v)
							->setMaxLength(255);
					else if ($v == 'country')
						${$com}->addSelect($v, 'eshopOrders.default.' . $v, $countries);
					else
						${$com}->addText($v, 'eshopOrders.default.' . $v)
							->setMaxLength(255);

					if ($v === 'idNumber' && EshopOrdersConfig::load('customer.allowValidateIdNumber')) {
						${$com}->addBool('validatedIdNumber', 'eshopOrders.default.validatedIdNumber');
					} else if ($v === 'vatNumber' && EshopOrdersConfig::load('customer.allowValidateVatNumber')) {
						${$com}->addBool('validatedVatNumber', 'eshopOrders.default.validatedVatNumber');
					}
				}
			}
		}

		if (EshopOrdersConfig::load('customer.allowMinimalOrderPrice', false)) {
			$form->addBool('allowMinimalOrderPrice', 'eshopOrders.customer.allowMinimalOrderPrice')
				->setDefaultValue(false);
			$form->addText('minimalOrderPrice', 'eshopOrders.customer.minimalOrderPrice')
				->setDefaultValue('0');
		}

		if (EshopOrdersConfig::load('customer.allowShowPricesWithoutVat', false)) {
			$form->addBool('showPricesWithoutVat', 'eshopOrders.customer.showPricesWithoutVat')
				->setDefaultValue(0);
		}

		$form->addBool('isActive', 'default.isActive')->setRequired();
		$form->addSaveCancelControl('saveControl');

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
		try {
			if (!$this->customer) {
				$this->getPresenter()->flashMessageWarning('eshopOrders.customerForm.noCreate');

				return false;
			}

			$customer     = $this->customer;
			$user         = $customer->getUser();
			$flashMessage = 'eshopOrders.customerForm.edited';

			$customer->setPhone($values->phone);
			$user->setName($values->name);
			$user->setLastname($values->lastname);
			$user->setEmail($values->email);
			if ($values->group) {
				$customer->setGroupCustomers($this->em->getReference(GroupCustomers::class, $values->group));
			} else {
				$customer->removeGroupCustomers();
			}

			if ($this->customersService->isAdmin($customer->getId()) && $values->isActive != $user->isActive()) {
				$this->getPresenter()->flashMessageWarning('eshopOrders.customerForm.noAdminDeactivate');
			} else {
				$user->disable(!$values->isActive);
			}

			if (!EshopOrdersConfig::load('allowAddressManagement')) {
				$addressDelivery = $customer->getAddressDelivery() ?: new CustomerAddress($customer);
				$addressInvoince = $customer->getAddressInvoice() ?: new CustomerAddress($customer);

				$customer->setAddressDelivery($addressDelivery);
				$customer->setAddressInvoice($addressInvoince);

				$fields = ['firstName', 'lastName', 'email', 'phone', 'company', 'street', 'city', 'postal', 'idNumber',
					'vatNumber'];
				if (EshopOrdersConfig::load('customer.showIdVatNumber', false))
					$fields[] = 'idVatNumber';
				$fields[] = 'country';

				foreach ($fields as $v) {
					foreach (['addressDelivery', 'addressInvoice'] as $com) {
						if ($v == 'country') {
							if ($values->$com->$v)
								$customer->$com->$v = $this->countries->getReference($values->$com->$v);
						} else {
							$customer->$com->$v = $values->$com->$v;
						}

						if ($v === 'idNumber' && EshopOrdersConfig::load('customer.allowValidateIdNumber')) {
							$customer->$com->validatedIdNumber = (int) $values->$com->validatedIdNumber;
						} else if ($v === 'vatNumber' && EshopOrdersConfig::load('customer.allowValidateVatNumber')) {
							$customer->$com->validatedVatNumber = (int) $values->$com->validatedVatNumber;
						}
					}
				}

				$this->em->persist($addressDelivery);
				$this->em->persist($addressInvoince);
			}

			$currentDisabledPayments = $customer->disabledPayments->toArray();
			foreach ($values->disabledPayments as $id) {
				if (!isset($currentDisabledPayments[$id]))
					$customer->disabledPayments->set($id, $this->payments->getReference($id));

				unset($currentDisabledPayments[$id]);
			}

			foreach ($currentDisabledPayments as $entity)
				$customer->disabledPayments->remove($entity->getId());

			$currentDisabledSpeditions = $customer->disabledSpeditions->toArray();
			foreach ($values->disabledSpeditions as $id) {
				if (!isset($currentDisabledSpeditions[$id]))
					$customer->disabledSpeditions->set($id, $this->speditions->getReference($id));

				unset($currentDisabledSpeditions[$id]);
			}

			foreach ($currentDisabledSpeditions as $entity)
				$customer->disabledSpeditions->remove($entity->getId());

			$this->em->persist($customer);
			$this->em->persist($user);

			if (UsersConfig::load('allowLoginAsAnotherUser')) {
				$otherLogins        = $form->getHttpData()['customerOtherLogins'] ?? [];
				$currentOtherLogins = $this->getOtherUsersForLogin();

				// set
				foreach (array_diff($otherLogins, $currentOtherLogins) as $id) {
					$ref          = $this->users->getReference($id);
					$ref->loginAs = $user;

					$this->em->persist($ref);
				}

				// remove
				foreach (array_diff($currentOtherLogins, $otherLogins) as $id) {
					$ref          = $this->users->getReference($id);
					$ref->loginAs = null;

					$this->em->persist($ref);
				}
			}

			$this->em->flush();

			if (EshopOrdersConfig::load('customer.allowShowPricesWithoutVat', false)) {
				$efShowPricesWithoutVat = EntityHelpers::setExtraField($customer, 'showPricesWithoutVat', (int) $values->showPricesWithoutVat);
				$this->em->persist($efShowPricesWithoutVat);
				$this->em->flush();
			}

			if (EshopOrdersConfig::load('customer.allowMinimalOrderPrice', false)) {
				$efAllowMinimalOrderPrice = EntityHelpers::setExtraField($customer, 'allowMinimalOrderPrice', $values->allowMinimalOrderPrice);
				$efMinimalOrderPrice      = EntityHelpers::setExtraField($customer, 'minimalOrderPrice', $values->minimalOrderPrice);

				$this->em->persist($efAllowMinimalOrderPrice);
				$this->em->persist($efMinimalOrderPrice);
				$this->em->flush();
			}

			$form->addCustomData('customerId', $customer->getId());
			$this->getPresenter()->flashMessageSuccess($flashMessage);
		} catch (\Exception $e) {
			$form->addError($e->getMessage());
			$this->redrawControl('form');

			return false;
		}

		return true;
	}

	public function setCustomer($id)
	{
		$this->customer = $customer = $this->customersService->get($id);

		if ($customer) {
			$o = $customer;
			$u = $o->getUser();

			$d = [
				'phone'              => $o->getPhone(),
				'name'               => $u->getName(),
				'lastname'           => $u->getLastname(),
				'email'              => $u->getEmail(),
				'isActive'           => $u->isActive(),
				'disabledPayments'   => $o->disabledPayments->getKeys(),
				'disabledSpeditions' => $o->disabledSpeditions->getKeys(),
			];

			if (!EshopOrdersConfig::load('allowAddressManagement')) {
				$fields = ['firstName', 'lastName', 'email', 'phone', 'company', 'street', 'city', 'postal', 'idNumber',
					'vatNumber'];
				if (EshopOrdersConfig::load('customer.showIdVatNumber', false))
					$fields[] = 'idVatNumber';
				$fields[] = 'country';

				foreach ($fields as $v) {
					foreach (['addressDelivery', 'addressInvoice'] as $com) {
						if (!$customer->$com)
							continue;

						if ($v == 'country') {
							$d[$com][$v] = $customer->$com->$v ? $customer->$com->$v->getId() : null;
						} else {
							$d[$com][$v] = $customer->$com->$v;
						}

						if ($v === 'idNumber' && EshopOrdersConfig::load('customer.allowValidateIdNumber')) {
							$d[$com]['validatedIdNumber'] = $customer->$com->validatedIdNumber;
						} else if ($v === 'vatNumber' && EshopOrdersConfig::load('customer.allowValidateVatNumber')) {
							$d[$com]['validatedVatNumber'] = $customer->$com->validatedVatNumber;
						}
					}
				}
			}

			$ef = $customer->getExtraFieldsValues();
			if (EshopOrdersConfig::load('customer.allowMinimalOrderPrice', false)) {
				foreach (['allowMinimalOrderPrice', 'minimalOrderPrice'] as $item) {
					$d[$item] = $ef[$item];
				}
			}

			if (EshopOrdersConfig::load('customer.allowShowPricesWithoutVat', false))
				$d['showPricesWithoutVat'] = $ef['showPricesWithoutVat'];

			$this['form']->setDefaults($d);

			if ($o->getGroupCustomers() && array_key_exists($o->getGroupCustomers()->getId(), $this['form']['group']->getItems()))
				$this['form']['group']->setDefaultValue($o->getGroupCustomers()->getId());
		} else
			$this->getPresenter()->error(null, IResponse::S404_NOT_FOUND);
	}

	protected function getOtherUsersForLogin(): array
	{
		if (!$this->customer)
			return [];

		if ($this->cOtherUsersForLogin === null) {
			$this->cOtherUsersForLogin = $this->customersService->getAllOtherUsersForLogin($this->customer->getUser()->getId());
		}

		return $this->cOtherUsersForLogin;
	}

	protected function createComponentAddressForm(): AddressForm
	{
		$control = $this->addressFormFactory->create();
		if ($this->getParameter('id'))
			$control->setAddress((int) $this->getParameter('id'));

		$control['form']->onSuccessSave[]         = function(BaseForm $form) {
			$this->getPresenter()->redrawControl('flashes');
			$this->redrawControl('addresses');
		};
		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form) {
			$this->getPresenter()->payload->hideModal = true;
			$this->getPresenter()->redrawControl('flashes');
			$this->redrawControl('addresses');
		};
		$control['form']['saveControl']->closeModalOnCancel();

		return $control;
	}
}
