<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Model\Statistics;

use Doctrine\ORM\QueryBuilder;
use EshopOrders\Model\Entities\Order;
use Core\Model\Entities\EntityManagerDecorator;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\Helpers\OrderHelper;
use EshopOrders\Model\Statistics\BaseStatistics;
use Nette\Utils\DateTime;
use Nette\Utils\FileSystem;
use Nette\Utils\Json;

class Statistics
{
	protected EntityManagerDecorator $em;
	protected BaseStatistics         $baseStatistics;

	protected ?array $cTodayStatusIds = null;

	protected ?array $cTodayCreated = null;

	public function __construct(EntityManagerDecorator $em, BaseStatistics $baseStatistics)
	{
		$this->em             = $em;
		$this->baseStatistics = $baseStatistics;
		$this->em->getConfiguration()->setSQLLogger(null);
	}

	public function getTodayStatusIds(string $status): array
	{
		if (!isset($this->cTodayStatusIds[$status])) {
			$this->cTodayStatusIds[$status] = [];

			foreach ($this->em->getRepository(OrderStatus::class)->createQueryBuilder('os')
				         ->select('IDENTITY(os.order) as order, os.id')
				         ->where('os.status = :status')
				         ->andWhere('os.created >= :today')
				         ->setParameters([
					         'status' => $status,
					         'today'  => (new DateTime())->setTime(0, 0, 0),
				         ])->getQuery()->getArrayResult() as $row)
				$this->cTodayStatusIds[$status][$row['order']] = $row;
		}

		return $this->cTodayStatusIds[$status];
	}

	public function getBaseQueryBuilder(): QueryBuilder
	{
		return $this->em->getRepository(Order::class)->createQueryBuilder('o', 'o.id')
			->addSelect('p, s, oi, oiSales, os, curr, od, pp, ss, site, inv, corDoc')
			->join('o.payment', 'p')
			->leftJoin('p.payment', 'pp')
			->join('o.spedition', 's')
			->leftJoin('s.spedition', 'ss')
			->innerJoin('o.site', 'site')
			->join('o.orderItems', 'oi')
			->leftJoin('oi.sales', 'oiSales')
			->innerJoin('o.orderStatuses', 'os')
			->leftJoin('o.currency', 'curr')
			->leftJoin('o.orderDiscounts', 'od')
			->leftJoin('o.addressInvoice', 'inv')
			->leftJoin('o.orderForCorrectiveTaxDocument', 'corDoc');
	}

	/**
	 * @return Order[]
	 */
	public function getTodayCreated(): array
	{
		if ($this->cTodayCreated === null) {
			$this->cTodayCreated = [];

			foreach ($this->getBaseQueryBuilder()
				         ->andWhere('os.created >= :today AND os.status = :status')
				         ->setParameters([
					         'status' => OrderStatus::STATUS_CREATED,
					         'today'  => (new DateTime())->setTime(0, 0, 0),
				         ])
				         ->groupBy('o.id')
				         ->getQuery()->getResult() as $order) {
				/** @var Order $order */
				$this->cTodayCreated[$order->getId()] = $order;
			}
		}

		return $this->cTodayCreated;
	}

	public function getTodayOrders(): array
	{
		$result = [
			'count' => 0,
			'price' => 0,
		];

		$canceled = $this->getTodayStatusIds(OrderStatus::STATUS_CANCELED);

		foreach ($this->getTodayCreated() as $order) {
			if (isset($canceled[$order->getId()]))
				continue;

			$result['count']++;
			$result['price'] += $order->getPriceItemsDiscount();
		}

		return $result;
	}

	public function getMonthlyFile(): string
	{
		return DATA_DIR . '/eshopOrders/stats/monthly.json';
	}

	public function getMonthly(bool $cached = true)
	{
		$file = $this->getMonthlyFile();
		if ($cached && file_exists($file) && DateTime::from(filemtime($file))->diff(new DateTime())->h <= 25) {
			$result = Json::decode(file_get_contents($file), Json::FORCE_ARRAY);
		} else {
			$result = $this->generateMonthly();
		}

		return $result;
	}

	public function generateMonthly(): array
	{
		$this->em->getConfiguration()->setSQLLogger(null);
		set_time_limit(3600);
		ini_set('memory_limit', '1G');
		$file = DATA_DIR . '/eshopOrders/stats/monthly.json';
		FileSystem::createDir(dirname($file));

		$canceled = [];
		$data     = [];
		$result   = [];

		$created = [];
		foreach ($this->em->getRepository(OrderStatus::class)->createQueryBuilder('os')
			         ->select('IDENTITY(os.status) as status, IDENTITY(os.order) as order, os.created')
			         ->where('os.status = :stat')
			         ->setParameter('stat', OrderStatus::STATUS_CREATED)
			         ->getQuery()->getArrayResult() as $row) {
			$created[$row['order']] = $row['created'];
		}

		foreach (array_chunk(array_keys($created), 500) as $chunk) {
			foreach ($this->baseStatistics->getCancelled($chunk) as $id) {
				$canceled[$id] = $id;
			}

			// Sleva objednavky
			foreach ($this->baseStatistics->getSales($chunk, true) as $id => $sales) {
				$data[$id]['discounts'] = $sales;
			}

			// Produkty
			foreach ($this->baseStatistics->getOrderItems($chunk) as $id => $orderItems) {
				$data[$id]['items'] = $orderItems;
			}

			// platba
			foreach ($this->baseStatistics->getPayments($chunk) as $id => $items) {
				$data[$id]['payments'] = $items;
			}

			// doprava
			foreach ($this->baseStatistics->getSpeditions($chunk) as $id => $items) {
				$data[$id]['speditions'] = $items;
			}
		}

		foreach ($data as $id => $vals) {
			if (isset($canceled[$id])) {
				continue;
			}

			$key = isset($created[$id]) ? $created[$id]->format('Y-m') : null;
			if (!$key) {
				continue;
			}

			$price = 0;
			foreach ($vals['items'] as $item) {
				$sales = $item['saleValue'] ? [OrderHelper::calculateSaleValue($item['saleType'], $item['saleValue'], $item['quantity'], $item['price'])] : [];
				$price += OrderHelper::getItemPriceTotal($item['price'], $item['quantity'], $sales);
			}

			foreach ($vals['payments'] as $item) {
				$price += $item['price'];
			}

			foreach ($vals['speditions'] as $item) {
				$price += $item['price'];
			}

			foreach ($vals['discounts'] as $discount) {
				$price -= $discount;
			}


			if (!isset($result[$key])) {
				$result[$key]         = $this->monthlyCreateRow($created[$id]);
				$result[$key]['date'] = $result[$key]['date']->format('Y-m');
			}

			$result[$key]['orders']++;
			$result[$key]['ordersPrice'] += $price;

			unset($data[$id]);
		}

		krsort($result, SORT_STRING);

		FileSystem::delete($file);
		FileSystem::write($file, Json::encode($result));

		return $result;
	}

	protected function monthlyCreateRow(\DateTime $dateTime)
	{
		return [
			'date'          => $dateTime,
			'orders'        => 0,
			'ordersPrice'   => 0,
			'canceled'      => 0,
			'canceledPrice' => 0,
		];
	}
}
