<?php declare(strict_types = 1);

namespace EshopOrders\ApiModule\Api\V1\Controllers;

use Apitte\Core\Annotation\Controller\Method;
use Apitte\Core\Annotation\Controller\Path;
use Apitte\Core\Http\ApiRequest;
use Apitte\Core\Http\ApiResponse;
use Apitte\Negotiation\Http\ArrayEntity;
use Core\Model\Application\AppState;
use Core\Model\Event\EventDispatcher;
use Core\Model\Countries;
use Core\Model\Entities\Site;
use Core\Model\Sites;
use EshopCatalog\Model\Entities\Product;
use EshopOrders\FrontModule\Model\Customers;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use EshopOrders\Model\Entities\InvoiceConfig;
use EshopOrders\Model\Entities\OrderAddress;
use EshopOrders\Model\Entities\OrderDiscount;
use EshopOrders\Model\Entities\OrderItemSale;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Event\OrderPayloadReturnEvent;
use EshopOrders\Model\Invoices;
use EshopOrders\Model\Receipts;
use EshopOrders\Model\Speditions;
use EshopOrders\Model\Statuses;
use EshopOrders\Model\Payments;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderItem;
use EshopOrders\Model\Entities\OrderPayment;
use EshopOrders\Model\Entities\OrderSpedition;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\Orders;
use EshopCatalog\Model\Config as EshopCatalogConfig;
use Core\Model\Entities\EntityManagerDecorator;
use EshopOrders\Model\Utils\Helpers;
use EshopStock\Model\Repository\SupplyProducts;
use Nette\Application\LinkGenerator;
use Nette\Localization\Translator;
use Nette\Utils\DateTime;
use Tracy\Debugger;


/**
 * @Path("/orders")
 */
class OrdersController extends BaseController
{
	protected EventDispatcher        $eventDispatcher;
	protected EntityManagerDecorator $em;
	protected Sites                  $sitesService;
	protected Orders                 $ordersService;
	protected Payments               $paymentsService;
	protected Speditions             $speditionsService;
	protected Statuses               $statusesService;
	protected Customers              $customersService;
	protected Countries              $countriesService;
	protected Translator             $translator;
	protected LinkGenerator          $linkGenerator;
	protected Invoices               $invoices;
	protected Receipts               $receipts;

	public function __construct(
		EntityManagerDecorator $entityManager,
		Translator             $translator,
		Countries              $countriesService,
		Sites                  $sitesService,
		Orders                 $ordersService,
		Customers              $customersService,
		Payments               $paymentsService,
		Speditions             $speditionsService,
		Statuses               $statusesService,
		EventDispatcher        $eventDispatcher,
		LinkGenerator          $linkGenerator,
		Invoices               $invoices,
		Receipts               $receipts
	)
	{
		$this->em                = $entityManager;
		$this->sitesService      = $sitesService;
		$this->ordersService     = $ordersService;
		$this->paymentsService   = $paymentsService;
		$this->speditionsService = $speditionsService;
		$this->statusesService   = $statusesService;
		$this->customersService  = $customersService;
		$this->countriesService  = $countriesService;
		$this->translator        = $translator;
		$this->eventDispatcher   = $eventDispatcher;
		$this->linkGenerator     = $linkGenerator;
		$this->invoices          = $invoices;
		$this->receipts          = $receipts;
	}

	/**
	 * @param ApiRequest  $request
	 * @param ApiResponse $response
	 *
	 * @return ApiResponse
	 *
	 * @Path("/create")
	 * @Method("POST")
	 * @throws \Doctrine\DBAL\ConnectionException
	 */
	public function create(ApiRequest $request, ApiResponse $response)
	{
		$data = $request->getJsonBody();

		try {
			$this->em->getConnection()->beginTransaction();

			$order = new Order($this->em->getReference(Site::class, $this->sitesService->getCurrentSite()->getIdent()));
			$order->setMessage($data['globalNote'] ?? '');
			$order->setAgreedTerms(true);

			AppState::setState('eshopOrdersPaymentMethod', 'eshopOrders.paymentMethod.checkout');
			$order->isPaid = 1;
			$order->paid   = new DateTime;
			$orderItems    = $this->convertJSONProductsToOrderItems($data['checkoutProducts'], $order);
			$isCorrective  = Helpers::isCorrectiveByOrderItems($orderItems);
			$order->setOrderItems($orderItems);
			$order->isCorrectiveTaxDocument = (int) $isCorrective;

			$orderPayment   = new OrderPayment($this->paymentsService->getReferenceByIdent($data['payment']), $order);
			$orderSpedition = new OrderSpedition($this->speditionsService->getReferenceByIdent($data['spedition']), $order);

			$statusCreated        = $this->statusesService->get(OrderStatus::STATUS_CREATED);
			$orderActionCreated   = new OrderStatus($order, $statusCreated);
			$statusSpedition      = $this->statusesService->get(OrderStatus::STATUS_SPEDITION);
			$orderActionSpedition = new OrderStatus($order, $statusSpedition);
			$statusFinished       = $this->statusesService->get(OrderStatus::STATUS_FINISHED);
			$orderActionFinished  = new OrderStatus($order, $statusFinished);

			$order->setPayment($orderPayment);
			$order->setSpedition($orderSpedition);
			$order->setOrderStatuses([$orderActionCreated, $orderActionFinished]);

			$addressInvoice  = $this->createAddress(new OrderAddress(OrderAddress::ADDRESS_INVOICE), $data['customer']['addressInvoice']);
			$addressDelivery = $this->createAddress(new OrderAddress(OrderAddress::ADDRESS_DELIVERY), $data['customer']['addressDelivery']);
			$order->setAddressInvoice($addressInvoice);
			$order->setAddressDelivery($addressDelivery);

			if ($data['customer']['customerId'] && $customer = $this->customersService->get((int) $data['customer']['customerId'])) {
				bdump($customer);
				$order->setCustomer($customer);
				$this->em->persist($customer);
			}

			$this->addGlobalSales($data['globalSales'], $order, $data['totalPrice']);


			/*if ($data['customer']['customerCreated'] && ($data['customer']['delivery']['email'] || $data['customer']['invoice']['email'])) {
				// Id vkladam jenom formalne - nemel by existovat
				$customer = $this->customersService->getOrCreateCustomer($data['customer']['customerId']);
				$customer->setAddressDelivery($this->createAddress(new CustomerAddress($customer), $data['customer']['delivery']));
				$customer->setAddressInvoice($this->createAddress(new CustomerAddress($customer), $data['customer']['delivery']));
				$this->em->persist($customer);
			}*/

			$event                 = new OrderEvent($order);
			$event->orderPayment   = $orderPayment;
			$event->orderSpedition = $orderSpedition;
			$event->addrDelivery   = $addressDelivery;
			$event->addrInvoice    = $addressInvoice;
			$event->orderItems     = $orderItems;
			$event->formData       = $data;

			$eshopStockSupplyProduct = class_exists('EshopStock\DI\EshopStockExtension') ? SupplyProducts::class : null;

			if ($eshopStockSupplyProduct !== null) {
				$eshopStockSupplyProduct::$assignOrderItemFlush = false;
			}
			
			$this->eventDispatcher->dispatch($event, 'eshopCheckout.orderBeforeSave');

			if (EshopOrdersConfig::load('receipt.enableUseNumericalSeries', false) && in_array($orderPayment->getIdent(), ['storeCard', 'storeCash'])) {
				$order->receiptIdent = $this->receipts->generateIdent($order);
			}

			$this->em->persist($order);
			$this->em->persist($orderSpedition);
			$this->em->persist($orderPayment);
			$this->em->persist($orderActionCreated);
			$this->em->persist($orderActionSpedition);
			$this->em->persist($orderActionFinished);

			$this->em->flush();

			if ($isCorrective) {
				$invoice = $this->invoices->getOneByOrder($order, true, InvoiceConfig::TYPE_CORRECTIVE);
				$order->setInvoice($invoice);
				$this->em->flush();
				$this->eventDispatcher->dispatch($event, 'eshopOrders.admin.correctiveTaxDocumentCreated');
			}

			$this->em->getConnection()->commit();

			$this->eventDispatcher->dispatch($event, 'eshopOrders.admin.orderOnSuccess');

			$result = [
				'success' => $order,
			];

			$orderPayloadReturnEvent = new OrderPayloadReturnEvent($result, $order);
			$this->eventDispatcher->dispatch($orderPayloadReturnEvent, 'eshopOrders.api.orderPayloadReturnEvent');

			return $response->withStatus(ApiResponse::S200_OK)->withEntity(ArrayEntity::from($orderPayloadReturnEvent->data));
		} catch (\Exception $e) {
			Debugger::log($e);
			if ($this->em->getConnection()->isTransactionActive()) {
				$this->em->getConnection()->rollBack();
			}

			return $response->withStatus(ApiResponse::S200_OK)->withEntity(ArrayEntity::from([
				'error' => $e->getMessage(),
				'data'  => $data,
			]));
		}
	}

	/**
	 * @param $JSONproducts
	 * @param $order
	 *
	 * @return array
	 * @throws \Doctrine\ORM\ORMException
	 */
	private function convertJSONProductsToOrderItems($JSONproducts, $order): array
	{
		$result       = [];
		$isCorrective = false;
		foreach ($JSONproducts as $item) {
			$product = $this->em->getReference(Product::class, $item['id']);
			$lang    = $this->translator->getLocale();
			$name    = $item['name'];

			if (EshopCatalogConfig::load('addManufacturerBeforeProductName', false) && $product->getManufacturer()) {
				$name = trim($product->getManufacturer()->name . ' ' . $name);
			}

			$price     = (float) $item['price'];
			$quantity  = (int) $item['count'];
			$orderItem = new OrderItem($product, $order);
			$orderItem->setQuantity(abs($quantity));
			$orderItem->addOrderItemText($lang);
			$orderItem->getOrderItemText($lang)->setName($name);
			if ($quantity < 0 || $isCorrective) { // pokud ma alespon jeden produkt zapornou kvantitu, pak se objednavka stava oprav. dan. dokladem
				$price        = -$price;
				$isCorrective = true;
			}

			$orderItem->setPrice($price);
			$orderItem->recyclingFee = $product->recyclingFee;
			$orderItem->setVatRate($item['vatRate']);
			$orderItem->setCode1($item['code1']);

			$this->addOrderItemsSales($item, $orderItem);

			$this->em->persist($orderItem);
			$result[] = $orderItem;
		}

		return $result;
	}

	private function addGlobalSales($globalSales, $order, $totalPrice)
	{
		foreach ($globalSales as $sale) {
			$saleEntity = new OrderDiscount('CHECKOUTSALE', $order);
			$saleEntity->setName($sale['name']);

			if (!empty($sale['saleCode'])) {
				$saleEntity->setCode($sale['saleCode']);
			}

			// procentualni sleva
			if ($sale['percentSale'] > 0) {
				$saleEntity->setValue((float) $sale['percentSale']);
				$saleEntity->setType(OrderDiscount::TYPE_PERCENT);
				$price = (100 / (100 - (float) $sale['percentSale'])) * $totalPrice; // cena pred uplatneni slevy
				$saleEntity->setPrice($price - $totalPrice);
			}

			// pevna sleva
			if ($sale['staticSale'] > 0) {
				$saleEntity->setValue((float) $sale['staticSale']);
				$saleEntity->setType(OrderDiscount::TYPE_FIX);
				$saleEntity->setPrice((float) $sale['staticSale']);
			}

			$order->addOrderDiscount($saleEntity);
			$this->em->persist($saleEntity);
		}
	}

	/**
	 * @param $item
	 * @param $orderItem
	 */
	private function addOrderItemsSales($item, $orderItem): void
	{

		// Pevna sleva na vsechny kusy
		if (($staticSale = $item['sale']['all']['staticSale']) > 0) {
			$sale = new OrderItemSale($staticSale, $orderItem, OrderItemSale::TYPE_FIX_ALL);
			$this->em->persist($sale);
		}

		// Procentuelni sleva na vsechny kusy
		if (($percentSale = $item['sale']['all']['percentSale']) > 0) {
			$sale = new OrderItemSale($percentSale, $orderItem, OrderItemSale::TYPE_PERCENT_ALL);
			$this->em->persist($sale);
		}

		// Pevna sleva na kus
		if (($staticSale = $item['sale']['one']['staticSale']) > 0) {
			$sale = new OrderItemSale($staticSale, $orderItem, OrderItemSale::TYPE_FIX_ONE);
			$this->em->persist($sale);
		}

		// Procentuelni sleva na kus
		if (($percentSale = $item['sale']['one']['percentSale']) > 0) {
			$sale = new OrderItemSale($percentSale, $orderItem, OrderItemSale::TYPE_PERCENT_ONE);
			$this->em->persist($sale);
		}
	}

	/**
	 * @param $address
	 * @param $addressData
	 *
	 * @return
	 */
	private function createAddress($address, $addressData)
	{

		$address->setFirstName($addressData['firstName'] ?? '');
		$address->setLastName($addressData['lastName'] ?? '');
		$address->setEmail($addressData['email'] ?? '');
		$address->setPhone($addressData['phone'] ?? '');
		$address->setStreet($addressData['street'] ?? '');
		$address->setCity($addressData['city'] ?? '');
		$address->setPostal($addressData['postal'] ?? '');
		if (!empty($addressData['country'])) {
			$address->setCountry($this->countriesService->getReference($addressData['country']));
		}
		$address->setCompany($addressData['company'] ?? '');
		$address->setIdNumber($addressData['idNumber'] ?? '');
		$address->setVatNumber($addressData['vatNumber'] ?? '');
		$this->em->persist($address);
		bdump($address);

		return $address;
	}

}
