<?php declare(strict_types = 1);

namespace EshopOrders\Components;

use Core\Model\Countries;
use Core\Model\Helpers\FormHelper;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopOrders\Model\Entities\Customer;
use EshopOrders\Model\Entities\CustomerAddress;
use EshopOrders\Model\EshopOrdersConfig;
use Nette\Utils\ArrayHash;

class AddressForm extends BaseControl
{
	/** @var int|null @persistent */
	public ?int $addressId = null;

	/** @var int|null @persistent */
	public ?int $customerId = null;

	protected ?CustomerAddress $address = null;

	protected Countries $countries;

	public function __construct(Countries $countries)
	{
		$this->countries = $countries;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$countries = $this->countries->getAllNameColumn();

		$form->addText('addressName', 'eshopOrders.customerAddress.addressName')
			->setMaxLength(255);
		$form->addText('firstName', 'eshopOrders.default.firstName')
			->setMaxLength(255);
		$form->addText('lastName', 'eshopOrders.default.lastName')
			->setMaxLength(255);
		$form->addText('email', 'eshopOrders.default.email')
			->setMaxLength(255);
		$form->addText('phone', 'eshopOrders.default.phone')
			->setMaxLength(255);
		$form->addText('street', 'eshopOrders.default.street')
			->setMaxLength(255);
		$form->addText('city', 'eshopOrders.default.city')
			->setMaxLength(255);
		$form->addText('postal', 'eshopOrders.default.postal')
			->setMaxLength(255);
		$form->addSelect('country', 'eshopOrders.default.country', $countries);
		$form->addText('company', 'eshopOrders.default.company')
			->setMaxLength(255);
		$form->addText('idNumber', 'eshopOrders.default.idNumber')
			->setMaxLength(255);
		$form->addText('vatNumber', 'eshopOrders.default.vatNumber')
			->setMaxLength(255);
		if (EshopOrdersConfig::load('customer.showIdVatNumber', false))
			$form->addText('idVatNumber', 'eshopOrders.customerAddress.idVatNumber')
				->setMaxLength(255);


		$form->addSaveCancelControl('saveControl');

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
		try {
			$customer = $this->em->getRepository(Customer::class)->find($this->customerId);

			if ($this->addressId) {
				$address      = $this->em->getRepository(CustomerAddress::class)->find($this->addressId);
				$flashMessage = 'eshopOrders.customerAddress.edited';
			} else {
				/** @var Customer $customer */
				$address      = new CustomerAddress($customer);
				$flashMessage = 'eshopOrders.customerAddress.created';
			}

			$address->country = $values->country ? $this->countries->getReference($values->country) : null;
			unset($values->country);
			FormHelper::fillEntityByValues($address, $values);

			$this->em->persist($address);
			$this->em->flush();

			$customer->getAddressesAvailable()->set($address->getId(), $address);

			$form->addCustomData('addressId', $address->getId());
			$this->getPresenter()->flashMessageSuccess($flashMessage);
		} catch (\Exception $e) {
			$form->addError($e->getMessage());
			$this->redrawControl('form');

			return false;
		}

		return true;
	}

	public function setAddress(int $addressId): void
	{
		$this->addressId = $addressId;
		/** @var CustomerAddress $address */
		$address = $this->em->getRepository(CustomerAddress::class)->find($addressId);

		if (!$addressId)
			$this->error();

		FormHelper::fillContainerByEntity($this['form'], $address);
	}
}
