<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Components\Customer;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopOrders\Components\AddressForm;
use EshopOrders\Components\IAddressFormFactory;
use EshopOrders\FrontModule\Model\Customers;
use EshopOrders\Model\Entities\Customer;
use EshopOrders\Model\Entities\CustomerAddress;

class AddressesList extends BaseControl
{
	protected int $userId;

	protected ?Customer $customer = null;

	protected Customers $customers;

	protected IAddressFormFactory $addressFormFactory;

	/** @var callable[] */
	public array $onAddressSave = [];

	public function __construct(int $userId, Customers $customers, IAddressFormFactory $addressFormFactory)
	{
		$this->userId             = $userId;
		$this->customers          = $customers;
		$this->addressFormFactory = $addressFormFactory;
	}

	public function render(): void
	{
		$customer = $this->getCustomer();
		if (!$customer)
			return;

		$this->template->addresses = $customer->getAddressesAvailable()->toArray();
		$this->template->render($this->getTemplateFile());
	}

	protected function getCustomer(): ?Customer
	{
		if ($this->customer === null)
			$this->customer = $this->customers->getByUserId($this->userId);

		return $this->customer;
	}

	/** ================================================================================================================
	 * ===========  Handle
	 */

	public function handleEdit(int $id): void
	{
		$this->template->modalTitle      = $this->t('eshopOrdersFront.customer.addressEdit');
		$this['addressForm']->addressId  = $id;
		$this['addressForm']->customerId = $this->getCustomer()->getId();
		$this->template->showModal       = true;
		$this->redrawControl('modal');
	}

	public function handleAdd(): void
	{
		$this->template->modalTitle      = $this->t('eshopOrdersFront.customer.addNewAddress');
		$this['addressForm']->customerId = $this->getCustomer()->getId();
		$this->template->showModal       = true;
		$this->redrawControl('modal');
	}

	public function handleRemove(int $id): void
	{
		$this->em->remove($this->em->getReference(CustomerAddress::class, $id));
		$this->em->flush();
		$this->getPresenter()->flashMessageSuccess('eshopOrdersFront.customer.addressRemoved');
		$this->getPresenter()->redrawControl('fleshed');
		$this->redrawControl('addresses');
	}

	/** ================================================================================================================
	 * ===========  Components
	 */

	protected function createComponentAddressForm(): AddressForm
	{
		$control = $this->addressFormFactory->create();

		if ($this->getParameter('id'))
			$control->setAddress((int) $this->getParameter('id'));
		$control['form']['saveControl']->removeComponent($control['form']['saveControl']['cancel']);
		$control['form']['saveControl']->removeComponent($control['form']['saveControl']['save']);
		$control['form']->setShowLangSwitcher(false);

		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form) {
			$this->getPresenter()->payload->hideModal = true;
			$this->template->showModal                = false;
			$this->redrawControl('modal');
			$this->getPresenter()->redrawControl('flashes');
			$this->redrawControl('addresses');
			$this->onAddressSave($this);
		};

		return $control;
	}
}
