<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Model;

use Core\Model\Application\AppState;
use Core\Model\Helpers\BaseService;
use EshopCatalog\FrontModule\Model\Dao\Product;
use EshopCatalog\Model\Config as EshopCatalogConfig;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\PaymentSpeditions;
use EshopSales\FrontModule\Model\OrderSales;
use Nette\DI\Container;
use Nette\Http\Session;
use Nette\Http\SessionSection;
use Nette\SmartObject;
use Users\Model\Security\User;

/**
 * Class CartHelper
 * @package EshopOrders\FrontModule\Model
 */
class CartHelper extends BaseService
{
	use SmartObject;

	const SESSION_KEY = 'eshopOrdersOrderForm';

	protected SessionSection    $sessionSection;
	protected Carts             $cartsService;
	protected Speditions        $speditionsService;
	protected Customers         $customers;
	protected Container         $container;
	protected User              $user;
	protected PaymentSpeditions $paymentSpedition;
	protected AppState          $appState;
	protected ?OrderSales       $orderSales = null;

	public function __construct(
		Carts             $carts,
		Speditions        $speditions,
		Customers         $customers,
		User              $user,
		Container         $container,
		Session           $session,
		PaymentSpeditions $paymentSpeditions,
		AppState          $appState
	)
	{
		$this->cartsService      = $carts;
		$this->speditionsService = $speditions;
		$this->customers         = $customers;
		$this->user              = $user;
		$this->container         = $container;
		$this->sessionSection    = $session->getSection(self::SESSION_KEY);
		$this->paymentSpedition  = $paymentSpeditions;
		$this->appState          = $appState;
	}

	// --------------- Doprava zdarma ------------

	public function countFreeSpedition()
	{
		$freeSpeditionPrice = $remainingPrice = null;

		if (EshopOrdersConfig::load('allowFreeFrom')) {
			$cart             = $this->cartsService->getCurrentCart();
			$speditionCountry = $this->getOrderFormData()['speditionCountry'] ?? 'CZ';

			$firstFreeSpedition = $this->paymentSpedition->getFirstFreeSpedition($speditionCountry);
			if ($firstFreeSpedition) {
				$freeSpeditionPrice = $firstFreeSpedition->getFreeFrom();

				$cart      = $this->cartsService->getCurrentCart();
				$cartPrice = $cart->getCartItemsPrice();

				if ($cartPrice > $freeSpeditionPrice) {
					$remainingPrice = 0;
				} else {
					$remainingPrice = $freeSpeditionPrice - $cartPrice;
				}
			}

			foreach ($cart->getCartItems() as $item) {
				if ($item->freeDelivery === true) {
					$freeSpeditionPrice = $item->getPrice(); //ne 0 -> to by znamenalo ze neni zdarma nikdy
					$remainingPrice     = 0;
				} else {
					if (!$firstFreeSpedition) {
						$freeSpeditionPrice = null;
						$remainingPrice     = null;
						break;
					}
				}
			}
		}

		return ['minPrice' => $freeSpeditionPrice, 'remaining' => $remainingPrice];
	}

	public function countFreeSpeditionMilestones()
	{
		$result = [];

		if (!EshopOrdersConfig::load('allowFreeFrom')) {
			return $result;
		}

		$cart             = $this->cartsService->getCurrentCart();
		$cartPrice        = $cart->getCartItemsPrice();
		$speditionCountry = $this->getOrderFormData()['speditionCountry'] ?? $this->appState->getCountry() ?: 'CZ';
		$speditions       = $this->paymentSpedition->getFirstFreeSpeditionByType($speditionCountry);

		foreach (['storePickup', 'pickup', 'address'] as $k) {
			$firstFreeSpedition = $speditions[$k];
			$freeSpeditionPrice = $remainingPrice = null;

			if ($firstFreeSpedition) {
				$freeSpeditionPrice = $firstFreeSpedition->getFreeFrom();

				if ($cartPrice > $freeSpeditionPrice) {
					$remainingPrice = 0;
				} else {
					$remainingPrice = $freeSpeditionPrice - $cartPrice;
				}
			}

			foreach ($cart->getCartItems() as $item) {
				if ($item->freeDelivery === true) {
					$freeSpeditionPrice = $item->getPrice(); //ne 0 -> to by znamenalo ze neni zdarma nikdy
					$remainingPrice     = 0;
					break;
				}
			}

			$result[$k] = ['minPrice' => $freeSpeditionPrice, 'remaining' => $remainingPrice];
		}

		return $result;
	}

	public function getOrderFormData(): array { return $this->sessionSection->orderFormData ?: []; }

	public function setOrderSales(OrderSales $orderSales): void
	{
		$this->orderSales = $orderSales;
	}

	public static function canAddToCart(Product $product): bool
	{
		if (!$product->getAvailability() || !$product->getAvailability()->canAddToCart()) {
			return false;
		}

		$quantity = $product->getQuantity() + $product->getQuantityExternal();

		return !EshopCatalogConfig::load('pseudoWarehouse')
			|| $quantity > 0
			|| (int) $product->unlimitedQuantity === 1
			|| (
				EshopCatalogConfig::load('allowWarehouseOverdraft')
				&& $product->getAvailability()->warehouseOverdraft
			);
	}
}

