<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Model\Subscribers;

use Core\Model\Event\EventDispatcher;
use Core\Model\Parameters;
use Core\Model\Settings;
use Core\Model\Sites;
use EshopCatalog\FrontModule\Model\Sellers;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\Event\EmailEvent;
use EshopOrders\Model\Helpers\QrGenerator;
use EshopOrders\Model\Payments;
use EshopOrders\Model\Utils\Helpers;
use Mpdf\Output\Destination;
use Nette\Bridges\ApplicationLatte\Template;
use Nette\Mail\SmtpMailer;
use Nette\Utils\Validators;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Core\Model\Mailing\TemplateFactory;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Invoices;
use EshopOrders\Model\Pdf\IInvoicePdfBuilderFactory;
use Exception;
use Nette\Http\Session;
use Nette\Http\SessionSection;
use Nette\Localization\ITranslator;
use Nette\Mail\IMailer;
use Nette\Mail\Message;
use Nette\Application\LinkGenerator;
use Tracy\Debugger;

class OrderMailerSubscriber implements EventSubscriberInterface
{
	/** @var ITranslator */
	protected $translator;

	/** @var TemplateFactory */
	protected $templateFactory;

	/** @var IMailer */
	protected $mailer;

	/** @var SessionSection */
	protected $orderSuccessMessages;

	/** @var Invoices */
	protected $invoices;

	/** @var IInvoicePdfBuilderFactory */
	protected $invoicePdfBuilderFactory;

	protected EventDispatcher $eventDispatcher;
	protected Sellers         $sellers;
	protected Payments        $paymentsService;
	protected QrGenerator     $qrGenerator;
	protected LinkGenerator   $linkGenerator;
	protected Settings        $settings;

	public function __construct(ITranslator     $translator, TemplateFactory $templateFactory, IMailer $mailer,
	                            Session         $session, Invoices $invoices, IInvoicePdfBuilderFactory $invoicePdfBuilderFactory,
	                            EventDispatcher $eventDispatcher, Sellers $sellers, Payments $payments, QrGenerator $qrGenerator,
	                            LinkGenerator   $linkGenerator, Settings $settings)
	{
		$this->translator               = $translator;
		$this->templateFactory          = $templateFactory;
		$this->orderSuccessMessages     = $session->getSection('eshopOrders/orderSuccessMessages');
		$this->invoices                 = $invoices;
		$this->invoicePdfBuilderFactory = $invoicePdfBuilderFactory;
		$this->eventDispatcher          = $eventDispatcher;
		$this->sellers                  = $sellers;
		$this->paymentsService          = $payments;
		$this->qrGenerator              = $qrGenerator;
		$this->linkGenerator            = $linkGenerator;
		$this->settings                 = $settings;

		if (EshopOrdersConfig::load('orderForm.enableSmtp', false) && Parameters::load('mailSender.smtp', null))
			$this->mailer = new SmtpMailer(Parameters::load('mailSender.smtp'));
		else
			$this->mailer = $mailer;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			'eshopOrders.orderOnSuccess'        => ['orderOnSuccess', 100],
			'eshopOrders.sendActualOrder'       => ['sendActualOrder', 100],
			'eshopOrders.sendInvoice'           => ['sendInvoice', 100],
			'eshopOrders.sendSurvey'            => ['sendSurvey', 100],
			'eshopOrders.sendRequestsForReview' => ['sendRequestsForReview', 100],
		];
	}

	/**
	 * @param OrderEvent $event
	 *
	 * @throws Exception
	 */
	public function orderOnSuccess(OrderEvent $event): void
	{
		if (!EshopOrdersConfig::load('orderForm.send'))
			return;

		try {
			$order = $event->order;
			$this->translator->setLocale($order->lang);
			Sites::$currentIdentOverride = $order->lang;

			$seller     = $this->sellers->getSellerForSite($order->site->getIdent());
			$file       = TEMPLATES_DIR . '/Front/default/EshopOrders/_emails/orderSend.latte';
			$originFile = __DIR__ . '/../../_emails/orderSend.latte';

			if (!file_exists($file))
				$file = $originFile;

			$siteIdent   = $order->site->getIdent();
			$sellerEmail = Helpers::getSellerEmail($seller, $order->site, $order->lang);
			$sellerName  = Helpers::getSellerName($seller, $siteIdent, $order->lang);

			if (!$sellerEmail)
				return;

			$template = $this->templateFactory->create($siteIdent, $order->lang);

			$subject = $this->translator->translate('eshopOrders.emails.subject', [
				'orderId'  => $order->getId(),
				'siteName' => $template->siteName,
			]);

			$template->setFile($file);
			$template->setParameters([
				'templateParts'  => __DIR__ . '/../../_emails/_parts.latte',
				'subject'        => $subject,
				'order'          => $order,
				'originTemplate' => $originFile,
				'canPayWithCard' => EshopOrdersConfig::load('sendPayLinkInEmail')
					&& in_array($order->getPaymentIdent(), ['card', 'essox'])
					&& $this->paymentsService->getByIdent('card', true),
				'qrCode'         => EshopOrdersConfig::load('emails.showQrPayment') && $seller && $order->getPaymentIdent() === 'transfer'
					? $this->qrGenerator->getQrCodeByOrderAndSeller($order, $seller)
					: null,
			]);

			$message = new Message();
			$message->setSubject($subject);
			$message->setFrom($sellerEmail, $sellerName ?? null);

			$termsAndConditionsPdf = $this->settings->get('eshopOrdersTermsAndConditionsPdf')[$this->translator->getLocale()] ?? null;
			if ($termsAndConditionsPdf && file_exists(WWW_DIR . $termsAndConditionsPdf)) {
				$message->addAttachment(
					$this->translator->translate('eshopOrdersFront.cart.termsAndConditions'),
					file_get_contents(WWW_DIR . $termsAndConditionsPdf),
				);
			}

			if ($seller->sendOrderToSeller)
				$message->addTo($sellerEmail, $sellerName ?? null);

			$bcc = EshopOrdersConfig::load('orderForm.bcc');
			if ($bcc) {
				foreach (explode(',', $bcc) as $v) {
					if ($v && Validators::isEmail(trim($v)))
						$message->addBcc(trim($v));
				}
			}

			$message->addTo($order->getAddressInvoice()->getEmail(), $order->getAddressInvoice()->getFirstName() . ' ' . $order->getAddressInvoice()->getLastName());

			$this->eventDispatcher->dispatch(new EmailEvent($template, $message, $order, OrderStatus::STATUS_CREATED, $order->getNewestOrderStatus()), 'eshopOrders.emailCreateOrder');

			$message->setHtmlBody($template->renderToString());
			$this->mailer->send($message);
		} catch (Exception $e) {
			Debugger::log('Order email failed - ' . $event->order->getId(), 'eshopOrderEmail');
			Debugger::log($e, 'eshopOrderEmail');
		}
	}

	/**
	 * @param OrderEvent $event
	 *
	 * @throws Exception
	 */
	public function sendActualOrder(OrderEvent $event): void
	{
		$order = $event->order;
		$this->translator->setLocale($order->lang);
		Sites::$currentIdentOverride = $order->lang;

		$siteIdent  = $order->site->getIdent();
		$seller     = $this->sellers->getSellerForSite($siteIdent);
		$file       = TEMPLATES_DIR . '/Front/default/EshopOrders/_emails/actualOrder.latte';
		$originFile = __DIR__ . '/../../_emails/actualOrder.latte';

		$sellerEmail = Helpers::getSellerEmail($seller, $order->site, $order->lang);
		$sellerName  = Helpers::getSellerName($seller, $siteIdent, $order->lang);

		if (!file_exists($file))
			$file = $originFile;

		if (!$sellerEmail)
			return;

		$template = $this->templateFactory->create($order->site->getIdent(), $order->lang);

		$subject = $this->translator->translate('eshopOrdersFront.emails.subjects.actualOrder', [
			'orderId'  => $order->getId(),
			'siteName' => $template->siteName,
		]);

		$template->setFile($file);
		$template->setParameters([
			'templateParts'  => __DIR__ . '/../../_emails/_parts.latte',
			'subject'        => $subject,
			'order'          => $order,
			'originTemplate' => $originFile,
			'canPayWithCard' => EshopOrdersConfig::load('sendPayLinkInEmail')
				&& in_array($order->getPaymentIdent(), ['card', 'essox'])
				&& $this->paymentsService->getByIdent('card', true),
			'qrCode'         => EshopOrdersConfig::load('emails.showQrPayment') && $seller && $order->getPaymentIdent() === 'transfer'
				? $this->qrGenerator->getQrCodeByOrderAndSeller($order, $seller)
				: null,
		]);

		$message = new Message();
		$message->setSubject($subject);
		$message->setFrom($sellerEmail, $sellerName ?? null);

		$message->addTo($order->getAddressInvoice()->getEmail(), $order->getAddressInvoice()->getFirstName() . ' ' . $order->getAddressInvoice()->getLastName());

		$this->eventDispatcher->dispatch(new EmailEvent($template, $message, $order, OrderStatus::STATUS_CREATED, $order->getNewestOrderStatus()), 'eshopOrders.emailActualStatus');

		$message->setHtmlBody($template->renderToString());
		$this->mailer->send($message);
	}

	/**
	 * @param OrderEvent $event
	 *
	 * @throws Exception
	 */
	public function sendSurvey(OrderEvent $event): void
	{
		$order = $event->order;
		$this->translator->setLocale($order->lang);
		Sites::$currentIdentOverride = $order->lang;

		$seller     = $this->sellers->getSellerForSite($order->site->getIdent());
		$file       = TEMPLATES_DIR . '/Front/default/EshopOrders/_emails/survey.latte';
		$originFile = __DIR__ . '/../../_emails/survey.latte';

		if (!file_exists($file))
			$file = $originFile;

		$sellerEmail = Helpers::getSellerEmail($seller, $order->site, $order->lang);
		$sellerName  = Helpers::getSellerName($seller, $order->site->getIdent(), $order->lang);

		if (!$sellerEmail)
			return;

		$template = $this->templateFactory->create($order->site->getIdent(), $order->lang);

		$subject = $this->translator->translate('eshopOrdersFront.emails.subjects.survey', [
			'siteName' => $template->siteName,
		]);

		$template->setFile($file);
		$template->setParameters([
			'templateParts'  => __DIR__ . '/../../_emails/_parts.latte',
			'subject'        => $subject,
			'order'          => $order,
			'originTemplate' => $originFile,
		]);

		$message = new Message();
		$message->setSubject($subject);
		$message->setFrom($sellerEmail, $sellerName ?? null);

		$message->addTo($order->getAddressInvoice()->getEmail(), $order->getAddressInvoice()->getFirstName() . ' ' . $order->getAddressInvoice()->getLastName());

		$this->eventDispatcher->dispatch(new EmailEvent($template, $message, $order, OrderStatus::STATUS_CREATED, $order->getNewestOrderStatus()), 'eshopOrders.emailSurvey');

		$message->setHtmlBody($template->renderToString());
		$this->mailer->send($message);
	}

	public function sendInvoice(OrderEvent $event): void
	{
		$order = $event->order;

		if (!EshopOrdersConfig::load('invoice.enable'))
			return;

		$template = $this->templateFactory->create($order->site->getIdent(), $order->lang);
		$this->setTemplateFile($template, 'invoice');

		$subject = $this->translator->translate('eshopOrdersFront.emails.subjects.invoice', [
			'orderId'  => $order->getId(),
			'siteName' => $template->siteName,
		]);

		$message = $this->prepareEmail($event, $template, $subject);

		$invoice    = $this->invoices->getOneByOrder($order);
		$pdfBuilder = $this->invoicePdfBuilderFactory->create($invoice);
		$message->addAttachment($pdfBuilder->getFileName(), $pdfBuilder->render(Destination::STRING_RETURN));
		$message->setHtmlBody($template->renderToString());
		$this->mailer->send($message);
	}

	public function sendRequestsForReview(OrderEvent $event): void
	{
		$order = $event->order;

		if (!EshopOrdersConfig::load('orderItemReviews.enable')) {
			return;
		}

		$template = $this->templateFactory->create($order->site->getIdent(), $order->lang);
		$this->setTemplateFile($template, 'requestForReview');

		$subject = $this->translator->translate('eshopOrdersFront.emails.subjects.review', [
			'orderId'  => $order->getId(),
			'siteName' => $template->siteName,
		]);

		$message        = $this->prepareEmail($event, $template, $subject);
		$template->link = $this->linkGenerator->link('EshopOrders:Front:OrderItemReviews:giveReviews', ['orderIdent' => $order->getIdent()]);

		$message->setHtmlBody($template->renderToString());
		$this->mailer->send($message);
	}

	protected function setTemplateFile(Template &$template, string $fileName): Template
	{
		$file       = TEMPLATES_DIR . '/Front/default/EshopOrders/_emails/' . $fileName . '.latte';
		$originFile = __DIR__ . '/../../_emails/' . $fileName . '.latte';

		if (!file_exists($file))
			$file = $originFile;

		$template->originTemplate = $originFile;
		$template->setFile($file);

		return $template;
	}

	protected function prepareEmail(OrderEvent $event, Template &$template, string $subject): ?Message
	{
		$order     = $event->order;
		$siteIdent = $order->site->getIdent();
		$seller    = $this->sellers->getSellerForSite($siteIdent);

		$sellerEmail = Helpers::getSellerEmail($seller, $order->site, $order->lang);
		$sellerName  = Helpers::getSellerName($seller, $siteIdent, $order->lang);

		if (!$sellerEmail)
			return null;

		$template->setParameters($template->getParameters() + [
				'templateParts'  => __DIR__ . '/../../_emails/_parts.latte',
				'subject'        => $subject,
				'order'          => $order,
				'canPayWithCard' => EshopOrdersConfig::load('sendPayLinkInEmail')
					&& in_array($order->getPaymentIdent(), ['card', 'essox'])
					&& $this->paymentsService->getByIdent('card', true),
			]);

		$message = new Message();
		$message->setSubject($subject);
		$message->setFrom($sellerEmail, $sellerName ?? null);

		$message->addTo($order->getAddressInvoice()->getEmail(), $order->getAddressInvoice()->getFirstName() . ' ' . $order->getAddressInvoice()->getLastName());

		return $message;
	}

}
