<?php declare(strict_types = 1);

namespace EshopOrders\Model\Entities;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;
use Core\Model\Entities\TId;
use EshopCatalog\Model\Entities\Product;

/**
 * @ORM\Table("eshop_orders__cart_item")
 * @ORM\Entity
 */
class CartItem
{
	use TId;

	/**
	 * Identifikator slozeny z id produktu, varianty, a pripadne dalsich parametru, ktere maji vliv na cenu
	 * @ORM\Column(name="ident", type="string", nullable=false)
	 */
	protected string $ident;

	/**
	 * @ORM\Column(name="product_id", type="integer", nullable=false)
	 */
	public int $productId;

	/**
	 * @ORM\Column(name="quantity", type="smallint", nullable=false, options={"default":1})
	 */
	public int $quantity;

	/**
	 * @ORM\ManyToOne(targetEntity="Cart", inversedBy="cartItems")
	 * @ORM\JoinColumn(name="cart_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	public Cart $cart;

	/**
	 * @ORM\Column(name="more_data", type="array", nullable=true)
	 */
	protected ?array $moreData = [];

	/**
	 * @var self[]|ArrayCollection
	 * @ORM\OneToMany(targetEntity="CartItem", mappedBy="parent")
	 */
	protected $children;

	/**
	 * @ORM\ManyToOne(targetEntity="CartItem", inversedBy="children")
	 * @ORM\JoinColumn(name="parent_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	protected ?CartItem $parent = null;

	public function __construct(string $ident, int $productId, Cart $cart, int $quantity = 1)
	{
		$this->ident     = $ident;
		$this->productId = $productId;
		$this->quantity  = $quantity;
		$this->cart      = $cart;
		$this->moreData  = [];
		$this->children  = new ArrayCollection();
		$this->cart->addCartItem($this); //Při vytvoření nové položky košíku je potřeba ji přidat rovnou do Entity košíku. Jelikož Proxy Entity ji po vytvoření neobsahuje
	}

	public function getIdent(): string { return $this->ident; }

	public function getProductId(): int { return $this->productId; }

	public function getMoreData(): array { return $this->moreData ?? []; }

	public function getMoreDataValue(string $key, $default = null)
	{
		return $this->moreData[$key] ?? $default;
	}

	public function setMoreData(array $data): self
	{
		if ($this->moreData['note']) {
			$data['note'] = $this->moreData['note'] . ($data['note'] ? "\n" . $data['note'] : '');
		}

		$this->moreData = $data;

		return $this;
	}

	public function setMoreDataValue(string $key, $value): self
	{
		if (!is_array($this->moreData))
			$this->moreData = [];

		$this->moreData[$key] = $value;

		return $this;
	}

	public function getChildren() { return $this->children; }

	public function getParent(): ?CartItem { return $this->parent; }

	public function setParent(CartItem $parent)
	{
		$this->parent = $parent;
		$parent->getChildren()->set($this->id, $this);
	}
}
