<?php declare(strict_types = 1);

namespace EshopOrders\Model\Entities;

use Core\Model\Helpers\Strings;
use Doctrine\ORM\Mapping as ORM;
use Core\Model\Entities\TId;
use EshopOrders\Model\Helpers\OrderHelper;

/** podobne jako order_item, ale samostatna tabulka
 * @ORM\Table("eshop_orders__order_payment")
 * @ORM\Entity
 */
class OrderPayment
{
	use TId;

	/**
	 * @var Payment
	 * @ORM\ManyToOne(targetEntity="Payment")
	 * @ORM\JoinColumn(name="payment_id", referencedColumnName="id", onDelete="SET NULL")
	 */
	public $payment;

	/**
	 * @var string
	 * @ORM\Column(name="name", type="string")
	 */
	protected $name;

	/**
	 * @var float
	 * @ORM\Column(name="price", type="decimal", precision=10, scale=2)
	 */
	protected $price;

	/**
	 * @var Order
	 * @ORM\OneToOne(targetEntity="Order", inversedBy="payment")
	 * @ORM\JoinColumn(name="order_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	public $order;

	/**
	 * @ORM\Column(name="vat_rate", type="smallint", nullable=true)
	 */
	protected ?int $vatRate = null;

	public function __construct(?Payment $payment, Order $order)
	{
		$this->payment = $payment;
		$this->name    = $payment ? $payment->getName() : '';
		if ($payment)
			$this->price = $payment->getPrice();
		$this->order = $order;
	}

	public function getIdent(): ?string { return $this->getPayment() ? $this->getPayment()->getIdent() : null; }

	public function getPaymentId(): ?int { return $this->payment ? $this->payment->getId() : null; }

	public function getPayment(): ?Payment { return $this->payment; }

	public function setPayment(Payment $payment): self
	{
		$this->payment = $payment;

		return $this;
	}

	public function getName(): string { return $this->name; }

	public function setName(string $name): self
	{
		$this->name = $name;

		return $this;
	}

	public function getPrice(bool $useCurrency = false): float
	{
		if ($this->getOrder() && $this->getOrder()->isZeroVat())
			return $this->getPriceWithoutVat($useCurrency);

		return $useCurrency
			? $this->getOrder()->calculateCurrencyPrice((float) $this->price)
			: (float) $this->price;
	}

	public function getPriceWithoutVat(bool $useCurrency = false): float
	{
		$price = $useCurrency
			? $this->getOrder()->calculateCurrencyPrice((float) $this->price)
			: (float) $this->price;

		return round(OrderHelper::removeVat($price, $this->vatRate), 2);
	}

	public function setPrice(float $price): self
	{
		$this->price = Strings::formatEntityDecimal($price);

		return $this;
	}

	public function getOrder(): Order { return $this->order; }

	public function setVatRate(?int $vatRate): void { $this->vatRate = $vatRate; }

	public function getVatRate(): int { return $this->getOrder()->isZeroVat() ? 0 : ($this->vatRate ?: 21); }
}
