<?php declare(strict_types = 1);

namespace EshopOrders\Model\Helpers;

use Core\Model\Helpers\Strings;
use Core\Model\SystemConfig;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderItemSale;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\EshopOrdersConfig;

class OrderHelper
{
	public static function canSendTrackingUrl(string $statusId, ?Order $order = null): bool
	{
		return in_array($statusId, [OrderStatus::STATUS_SPEDITION]);
	}

	public static function calculateCurrencyPrice(float $price, float $rate, int $decimals): float
	{
		return round($price * (1 / $rate), $decimals);
	}

	public static function removeVat(float $price, ?int $vat): float
	{
		if ((int) $vat === 0) {
			return $price;
		}

		return round($price / (1 + ($vat / 100)), 10);
	}

	public static function checkCountryVatRate(int $vatRate, string $country, bool $validatedVatNumber, ?string $idNumber = null, ?string $vatNumber = null): int
	{
		$country = mb_strtolower($country);
		if ($vatNumber && $validatedVatNumber && in_array($country, EshopOrdersConfig::load('invoice.zeroVatIfVatNumberValidatedFor'))) {
			return 0;
		}

		if ($country === 'sk' && $vatRate === 21 && EshopOrdersConfig::load('order.setSkVat20if21')) {
			return 20;
		}

		return $vatRate;
	}

	public static function getItemPriceWithoutVat(float $price, int $vatRate): float
	{
		return self::removeVat($price, $vatRate);
	}

	public static function getItemPriceTotalWithoutVat(float $price, int $quantity, int $vatRate, array $sales = []): float
	{
		$price = self::removeVat($price, $vatRate) * $quantity;

		foreach ($sales as $sale)
			$price -= $sale;

		return $price;
	}

	public static function getItemPriceTotal(float $price, int $quantity, array $sales = []): float
	{
		$price = $price * $quantity;

		foreach ($sales as $sale)
			$price -= $sale;

		return $price;
	}

	public static function calculateSaleValue(string $type, $value, int $quantity, float $basePrice)
	{
		switch ($type) {
			case OrderItemSale::TYPE_FIX_ALL:
				return $value;
			case OrderItemSale::TYPE_FIX_ONE:
				return $value * $quantity;
			case OrderItemSale::TYPE_PERCENT_ALL:
				return (($basePrice * $quantity) / 100) * $value;
			case OrderItemSale::TYPE_PERCENT_ONE:
				return (($basePrice / 100) * $value) * $quantity;
		}

		return $value;
	}

	public static function calculatePriceWithoutVatWithSales(float $totalPrice, float $totalPriceAfterSale, array $dphValues): float
	{
		$result = 0;

		foreach ($dphValues as $dph => $value)
			$result += (($totalPriceAfterSale * ($value / $totalPrice)) / (100 + $dph)) * 100;

		return $result;
	}

	public static function calculateVatWithSales(float $totalPrice, float $totalSale, array $dphValues): array
	{
		$result = [];

		foreach ($dphValues as $vat => $value) {
			$price                      = round($totalSale * ($value / $totalPrice), 2);
			$result[$vat]['total']      = -$price;
			$result[$vat]['withoutVat'] = -round($price / (($vat / 100) + 1), 2);
		}

		return $result;
	}

	public static function isOtherCountryVatFilled(?string $country, ?string $vat): bool
	{
		return $country && mb_strtoupper($country) !== SystemConfig::load('defaultCountry') && $vat;
	}

	public static function roundSkCod(float $price): float
	{
		$price = round($price, 2);

		$price *= 100;

		$lastNumber = substr((string) $price, -1);
		if ($lastNumber < 5) {
			$price = ceil($price / 5) * 5;
		} else {
			$price = round($price / 5) * 5;
		}

		$price /= 100;

		return $price;
	}
}
