<?php declare(strict_types = 1);

namespace EshopOrders\Model;

use Core\Model\Event\EventDispatcher;
use Core\Model\Helpers\BaseEntityService;
use Core\Model\Mailing\MailBuilderFactory;
use Core\Model\Mailing\TemplateFactory;
use Core\Model\Sites;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Query;
use Doctrine\ORM\Query\Expr\Join;
use Doctrine\ORM\QueryBuilder;
use EshopCatalog\FrontModule\Model\Sellers;
use EshopCatalog\Model\Entities\Availability;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\SellerInSite;
use EshopOrders\FrontModule\Model\Dao\CartItem;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderFlag;
use EshopOrders\Model\Entities\OrderItem;
use EshopOrders\Model\Entities\OrderItemGift;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\Helpers\OrderHelper;
use EshopOrders\Model\Utils\Helpers;
use Nette\Localization\ITranslator;
use Nette\Utils\DateTime;
use Tracy\Debugger;

/**
 * class Orders
 * @package EshopOrders\Model
 *
 * @method Order|object|null getReference($id)
 * @method Order[]|null getAll()
 * @method Order|null get($id)
 */
class Orders extends BaseEntityService
{
	protected $entityClass = Order::class;

	/** @var EventDispatcher|null @inject */
	public ?EventDispatcher $eventDispatcher = null;

	/** @var TemplateFactory|null @inject */
	public ?TemplateFactory $templateFactory = null;

	/** @var ITranslator|null @inject */
	public ?ITranslator $translator = null;

	/** @var MailBuilderFactory|null @inject */
	public ?MailBuilderFactory $mailer = null;

	/** @var Sellers|null @inject */
	public ?Sellers $sellers = null;

	/** @var Sites|null @inject */
	public ?Sites $sites = null;

	/** Vrati seznam objednavek podle zakaznika
	 * @return ArrayCollection|Order[]
	 */
	public function getByCustomer($customer = null)
	{
		if (is_null($customer)) {
			return new ArrayCollection();
		}

		$orderRawQuery = $this->getEr()->createQueryBuilder('o', 'o.id')
			->addSelect('pay, sped, inv, items, itemSales, itemChildren, discounts, curr, stat, invo, deli')
			->andWhere('o.customer = :customer')
			->andWhere('o.site = :site')
			->setParameters([
				'customer' => $customer,
				'site'     => $this->sites->getCurrentSite()->getIdent(),
			])
			->leftJoin('o.payment', 'pay')
			->leftJoin('o.spedition', 'sped')
			->leftJoin('o.invoice', 'inv')
			->leftJoin('o.orderItems', 'items')
			->leftJoin('items.sales', 'itemSales')
			->leftJoin('items.children', 'itemChildren')
			->leftJoin('o.orderDiscounts', 'discounts')
			->leftJoin('o.currency', 'curr')
			->leftJoin('o.orderStatuses', 'stat')
			->leftJoin('o.addressInvoice', 'invo')
			->leftJoin('o.addressDelivery', 'deli')
			->orderBy('o.id', 'DESC');

		return $orderRawQuery->getQuery()->getResult();
	}

	public function getFullOrder(int $id): ?Order
	{
		return $this->getEr()->createQueryBuilder('o')
			->addSelect('cus, pay, sped, items, discounts, deli, invo, stat, flags, site, gifts, curr')
			->leftJoin('o.customer', 'cus')
			->leftJoin('o.payment', 'pay')
			->leftJoin('o.spedition', 'sped')
			->leftJoin('o.orderItems', 'items')
			->leftJoin('o.orderDiscounts', 'discounts')
			->leftJoin('o.addressDelivery', 'deli')
			->leftJoin('o.addressInvoice', 'invo')
			->leftJoin('o.orderStatuses', 'stat')
			->leftJoin('o.orderFlags', 'flags')
			->leftJoin('o.site', 'site')
			->leftJoin('o.gifts', 'gifts')
			->leftJoin('o.currency', 'curr')
			->andWhere('o.id = :id')
			->setParameter('id', $id)
			->getQuery()->getOneOrNullResult();
	}

	public function getByIdent($ident): ?Order
	{
		if (is_null($ident)) {
			return null;
		}
		$orderQuery = $this->getEr()->createQueryBuilder('o', 'o.id');
		$orderQuery->andWhere('o.ident = :ident')->setParameter('ident', $ident)
			->setMaxResults(1);

		return $orderQuery->getQuery()->getOneOrNullResult();
	}

	/**
	 * @param CartItem[] $itemsCart
	 * @param Order      $order
	 *
	 * @return array
	 * @throws \Doctrine\ORM\ORMException
	 *
	 */
	public function fillOrderItems($itemsCart, $order)
	{
		/** @var OrderItem[] $items */
		$items     = [];
		$cartItems = [];
		foreach ($itemsCart as $ci) {
			$cartItems[] = $ci;

			foreach ($ci->getChilds() as $child)
				$cartItems[] = $child;
		}

		foreach ($cartItems as $ci) {
			$product = $ci->getProduct();
			$item    = new OrderItem($product ? $this->em->getReference(Product::class, $product->getId()) : null, $order);

			$item->setQuantity((int) $ci->getQuantity());

			$price = (float) ($product ? $product->priceInBaseCurrency : $ci->priceInBaseCurrency);
			if ($ci->getData('extraPrice')) {
				$price += (float) $ci->getData('extraPrice');
			}
			$item->setPrice($price);

			if ($product) {
				$item->recyclingFee = $product->getRecyclingFee();
				$addrInv            = $order->getAddressInvoice();
				if ($addrInv && $addrInv->getCountry()) {
					$vatRate = OrderHelper::checkCountryVatRate(
						(int) $product->getVatRate(),
						$addrInv->getCountry()->getId(),
						false,
						$addrInv->getIdNumber(),
						$addrInv->getVatNumber(),
					);
				} else {
					$vatRate = (int) $product->getVatRate();
				}

				$item->setVatRate($vatRate);
				$item->setCode1($product->getCode1());

				if ($product->getAvailability() && $product->getAvailability()->getIdent() === Availability::PREORDER) {
					$item->setMoreDataValue('isPreorder', true);

					/** @var Product $productEntity */
					$productEntity = $this->em->getRepository(Product::class)->find($product->getId());
					if ($productEntity) {
						$productEntity->setMoreDataValue('lastStatusPreorder', true);
						$this->em->persist($productEntity);
					}
				}
			}

			$item->addOrderItemText($this->translator->getLocale());
			$item->getOrderItemText($this->translator->getLocale())->setName($ci->title);

			if ($ci->discountDisabled)
				$item->setMoreDataValue('discountDisabled', true);

			$note = [];
			if ($ci->getData('staticNote'))
				$note[] = $ci->getData('staticNote');
			if ($ci->getData('note'))
				$note[] = $ci->getData('note');

			if ($note)
				$item->setMoreDataValue('note', implode("\n", $note));

			foreach ($ci->getGifts() as $gift) {
				// TODO možno upravit protože OrderItem by měl obshovat referenci na produkt
				$itemGift        = new OrderItemGift($item, $this->em->getReference(Product::class, $gift->getProductId()), $gift->getName());
				$itemGift->code1 = $gift->code1;
				$itemGift->ean   = $gift->ean;

				$this->em->persist($itemGift);
			}

			if ($ci->parentId) {
				$item->setParent($items[$ci->parentId]);
			}

			$this->em->persist($item);
			$items[$ci->getId()] = $item;
		}

		return $items;
	}

	public function setPaid(array $ids)
	{
		$this->em->beginTransaction();
		try {
			foreach ($ids as $id) {
				$e         = $this->getReference((int) $id);
				$e->isPaid = 1;

				$this->em->persist($e);
			}
			$this->em->flush();
			$this->em->commit();
		} catch (\Exception $e) {
			if ($this->em->getConnection()->isTransactionActive())
				$this->em->rollback();
			Debugger::log($e);

			return false;
		}

		return true;
	}

	public function proccessSurveysBulk()
	{
		ini_set('memory_limit', '4g');
		$config    = EshopOrdersConfig::load('sendSurveyAfterOrder');
		$bulkLimit = EshopOrdersConfig::load('surveyBulkLimit') ?: 20;

		if ($config === false)
			$this->terminate();

		$qb       = $this->getOrdersWithoutSurveySent($config, (int) $bulkLimit);
		$iterator = $qb->getQuery()->iterate();

		$i = 1;
		while (($row = $iterator->next()) !== false) {
			/** @var Order $order */
			$order = array_values($row)[0];
			$this->sendSurvey($order);
			$i++;

			if ($i % 20 === 0) {
				$this->em->clear();
				gc_collect_cycles();
			}
		}
	}

	public function sendSurvey(Order $order)
	{
		$this->eventDispatcher->dispatch(new OrderEvent($order), 'eshopOrders.sendSurvey');

		$flag = new OrderFlag(OrderFlag::TYPE_SURVEY, '1', $order);
		$order->addFlag($flag);

		$this->em->persist($flag);
		$this->em->flush();
	}

	public function sendInfo(Order $order, string $textSubject, string $message): bool
	{
		try {
			$file       = TEMPLATES_DIR . '/Front/default/EshopOrders/_emails/customerInfo.latte';
			$originFile = __DIR__ . '/../FrontModule/_emails/customerInfo.latte';

			$this->translator->setLocale($order->lang);
			Sites::$currentIdentOverride = $order->lang;

			$mailer    = $this->mailer->create();
			$siteIdent = $order->site->getIdent();
			$mailer->setTemplate($this->templateFactory->create($siteIdent, $order->lang));
			$seller = $this->sellers->getSellerForSite($siteIdent);

			$sellerEmail = Helpers::getSellerEmail($seller, $order->site, $order->lang);
			$sellerName  = Helpers::getSellerName($seller, $siteIdent, $order->lang);

			if (!file_exists($file))
				$file = $originFile;

			$subject = $this->translator->translate('eshopOrders.emails.subjectCustomerInfo',
				['orderId' => $order->getId(), 'siteName' => $mailer->getTemplate()->siteName]);

			$mailer->setTemplateFile($file);
			$mailer->setParameters([
				'templateParts'  => __DIR__ . '/../FrontModule/_emails/_parts.latte',
				'subject'        => $subject,
				'textSubject'    => $textSubject,
				'order'          => $order,
				'message'        => $message,
				'orderId'        => $order->getId(),
				'originTemplate' => $originFile,
			]);

			$mailer->setFrom($sellerEmail, $sellerName ?? null);
			$mailer->addTo($order->getAddressInvoice()->getEmail(), $order->getAddressInvoice()->getName());
			$mailer->send();
		} catch (\Exception $e) {
			bdump($e);
			Debugger::log($e, 'eshopOrders_sendInfo');

			return false;
		}

		return true;
	}

	/**
	 * @param int $daysWait
	 * @param int $bulkLimit
	 *
	 * @return QueryBuilder
	 */
	public function getOrdersWithoutSurveySent(int $daysWait, int $bulkLimit)
	{
		$date    = (new DateTime())->modify('-' . $daysWait . ($daysWait === 1 ? 'day' : ' days'));
		$maxDate = $date->modifyClone('-2 months');

		return $this->getEr()->createQueryBuilder('o')
			->addSelect('invo')
			->distinct()->innerJoin('o.orderStatuses', 'stat', Join::WITH, 'stat.status = :status AND stat.created <= :date AND stat.created >= :maxDate')
			->leftJoin('o.addressDelivery', 'deli')
			->innerJoin('o.addressInvoice', 'invo', Join::WITH, 'invo.email IS NOT NULL AND invo.email != \'\'')
			->distinct()->leftJoin('o.orderFlags', 'flags', Join::WITH, 'flags.type = :type')
			->andWhere('flags.id IS NULL')
			->setParameters([
				'status'  => OrderStatus::STATUS_FINISHED,
				'date'    => $date,
				'type'    => OrderFlag::TYPE_SURVEY,
				'maxDate' => $maxDate,
			])
			->setMaxResults($bulkLimit)
			->orderBy('o.id', 'DESC');
	}

	/**
	 * @param int|null           $sellerId
	 * @param \DateTimeInterface $from
	 * @param \DateTimeInterface $to
	 *
	 * @return Order[][]
	 */
	public function getByDateAndSeller(?int $sellerId = null, \DateTimeInterface $from, \DateTimeInterface $to): array
	{
		$result = [];
		$sites  = [];
		$ids    = [];

		$qb = $this->em->createQueryBuilder()
			->select('IDENTITY(sis.site) as site, IDENTITY(sis.seller) as seller')
			->from(SellerInSite::class, 'sis');

		if ($sellerId !== null) {
			$qb->where('sis.seller = :seller')
				->setParameter('seller', $sellerId);
		}

		foreach ($qb->getQuery()->getArrayResult() as $row) {
			$sites[$row['site']] = $row['seller'];
		}

		foreach ($this->em->createQueryBuilder()
			         ->select('o.id')
			         ->from(OrderStatus::class, 'os')
			         ->innerJoin('os.order', 'o', Join::WITH, 'o.site IN (:sites)')
			         ->andWhere('os.status = :status')
			         ->andWhere('os.created >= :from')
			         ->andWhere('os.created <= :to')
			         ->setParameters([
				         'status' => OrderStatus::STATUS_CREATED,
				         'sites'  => array_keys($sites),
				         'from'   => $from,
				         'to'     => $to,
			         ])
			         ->groupBy('o.id')->getQuery()->getArrayResult() as $row) {
			$ids[] = $row['id'];
		}

		foreach ($this->em->createQueryBuilder()
			         ->select('o, cus, op, os, del, inv, cur, cor')
			         ->from(Order::class, 'o')
			         ->leftJoin('o.customer', 'cus')
			         ->leftJoin('o.payment', 'op')
			         ->leftJoin('o.spedition', 'os')
			         ->leftJoin('o.addressDelivery', 'del')
			         ->leftJoin('o.addressInvoice', 'inv')
			         ->leftJoin('o.currency', 'cur')
			         ->leftJoin('o.orderForCorrectiveTaxDocument', 'cor')
			         ->andWhere('o.id IN (:ids)')
			         ->setParameters([
				         'ids' => $ids,
			         ])
			         ->groupBy('o.id')
			         ->orderBy('o.id', 'ASC')
			         ->getQuery()
			         ->getResult() as $row) {
			/** @var Order $row */
			$result[$sites[$row->site->getIdent()]][$row->getId()] = $row;
		}

		return $result;
	}

	public function validateIdNumber(Order $order, int $val = 1): void
	{
		$order->getAddressInvoice()->validatedIdNumber = $val;
		$this->em->persist($order->getAddressInvoice());
		$this->em->flush();
	}

	public function validateVatNumber(Order $order, int $val = 1): void
	{
		$order->getAddressInvoice()->validatedVatNumber = $val;
		$this->em->persist($order->getAddressInvoice());
		$this->em->flush($order->getAddressInvoice());
	}
}
