<?php declare(strict_types = 1);

namespace EshopOrders\Model;

use Core\Model\Countries;
use Core\Model\Event\Event;
use Core\Model\Event\EventDispatcher;
use Core\Model\Helpers\BaseEntityService;
use Core\Model\Helpers\Strings;
use Core\Model\Settings;
use Core\Model\Sites;
use Core\Model\SystemConfig;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\ProductSpedition;
use EshopOrders\FrontModule\Model\Customers;
use EshopOrders\Model\Entities\PaymentSpedition;
use Doctrine\ORM\Query\Expr\Join;
use EshopOrders\Model\Helpers\EshopOrdersCache;
use Nette\Caching\Cache;
use Users\Model\Security\User;
use EshopOrders\FrontModule\Model\Speditions;
use EshopOrders\FrontModule\Model\Payments;
use EshopOrders\FrontModule\Model\Dao;

/**
 * class PaymentSpeditions
 * @package EshopOrders\Model
 *
 * @method PaymentSpedition|object|null getReference($id)
 * @method PaymentSpedition|null get($id)
 */
class PaymentSpeditions extends BaseEntityService
{
	protected $entityClass = PaymentSpedition::class;

	public ?Dao\Cart $cart = null;

	public Sites               $sitesService;
	protected Countries        $countries;
	protected User             $user;
	protected Customers        $customers;
	protected Speditions       $speditions;
	protected Payments         $payments;
	protected Settings         $settings;
	protected EshopOrdersCache $eshopOrdersCache;
	protected EventDispatcher  $eventDispatcher;

	/** @var Dao\PaymentSpedition[]|null */
	protected ?array $cPublished = null;

	public function __construct(
		Sites            $sitesService,
		User             $user,
		Customers        $customers,
		Countries        $countries,
		Speditions       $speditions,
		Payments         $payments,
		Settings         $settings,
		EshopOrdersCache $eshopOrdersCache,
		EventDispatcher  $eventDispatcher
	)
	{
		$this->sitesService     = $sitesService;
		$this->countries        = $countries;
		$this->user             = $user;
		$this->customers        = $customers;
		$this->speditions       = $speditions;
		$this->payments         = $payments;
		$this->settings         = $settings;
		$this->eshopOrdersCache = $eshopOrdersCache;
		$this->eventDispatcher  = $eventDispatcher;
	}

	public function setPosition($id, $position)
	{
		if ($item = $this->get($id)) {
			$item->setPosition($position);
			$this->em->persist($item);
			$this->em->flush();

			return true;
		}

		return false;
	}

	public function setPublish($id, $state)
	{
		if ($item = $this->getReference($id)) {
			$item->isPublished = $state;
			$this->em->persist($item);
			$this->em->flush();

			return true;
		}

		return false;
	}

	/**
	 * @return Dao\PaymentSpedition[]
	 */
	public function getAllPublished(string $site = null): array
	{
		if ($site || $this->cPublished === null || EshopOrdersConfig::load('paymentSpeditions.frontDisableLoadingOptimalization', false)) {
			$this->cPublished   = [];
			$cart               = $this->cart;
			$payments           = $this->payments->getAllPublished();
			$speditions         = $this->speditions->getAllPublished();
			$disabledSpeditions = [];

			$this->eventDispatcher->dispatch(new Event(['payments' => &$payments, 'speditions' => &$speditions,
			                                            'cart'     => $cart]), self::class . '::beforeLoadPublished');

			if ($cart && Config::load('product.allowModifySpedition')) {
				$productsIds = [];
				foreach ($cart->getCartItems() as $item)
					if ($item->getProductId())
						$productsIds[] = $item->getProductId();

				if ($productsIds) {
					foreach ($this->em->createQueryBuilder()
						         ->select('IDENTITY(ps.spedition) as sped')
						         ->from(ProductSpedition::class, 'ps')
						         ->where('ps.product IN (:ids)')
						         ->setParameter('ids', $productsIds)->getQuery()->getArrayResult() as $row) {
						$disabledSpeditions[$row['sped']] = true;
					}
				}
			}

			$qb = $this->getEr()->createQueryBuilder('ps')
				->select('ps.id as id, p.id as payment, s.id as spedition, GROUP_CONCAT(c.id) as countries, ps.freeFromDate, ps.freeToDate, ps.currency')
				->andWhere('ps.isPublished = :isPublished')
				->innerJoin('ps.countries', 'c')
				->innerJoin('ps.payment', 'p')
				->innerJoin('ps.spedition', 's')
				->addOrderBy('s.position', 'ASC')
				->addOrderBy('p.position', 'ASC')
				->setParameters([
					'isPublished' => 1,
				])
				->groupBy('ps.id');


			if (EshopOrdersConfig::load('paymentSpeditions.allowCustomerGroups', false)) {
				$qb->leftJoin('ps.customerGroups', 'custG')
					->addSelect('GROUP_CONCAT(custG.id) as customerGroups');
			}

			if ($cart && $cart->hasPreorderProduct() && !empty($this->settings->get('eshopCatalogAllowedPaymentsForPreorder', []))) {
				$qb->andWhere('p.id IN (:pIds)')
					->setParameter('pIds', $this->settings->get('eshopCatalogAllowedPaymentsForPreorder', []));
			}

			if (count($this->sitesService->getSites()) > 1) {
				$qb->innerJoin('ps.sites', 'sites', Join::WITH, 'sites.ident = :site')
					->setParameter('site', $site ?? $this->sitesService->getCurrentSite()->getIdent());
			}

			if (EshopOrdersConfig::load('paymentSpeditions.allowCustomerGroups', false) && $this->user->isLoggedIn()) {
				$customer = $this->customers->getByUser($this->user->getIdentity());
			} else {
				$customer = null;
			}

			if (!empty(SystemConfig::load('siteAllowedCountries', []))) {
				$domain           = $this->sitesService->getCurrentSite()->getCurrentDomain();
				$allowedCountries = SystemConfig::load('siteAllowedCountries.' . $domain->site->getIdent() . '.' . $domain->getLang());

				if (!empty($allowedCountries)) {
					$qb->andWhere('c.id IN (:cIds)')
						->setParameter('cIds', $allowedCountries);
				}
			}

			foreach ($qb->getQuery()->getArrayResult() as $ps) {
				if ($ps['customerGroups'] && EshopOrdersConfig::load('paymentSpeditions.allowCustomerGroups', false)) {
					$groups        = array_unique(explode(',', $ps['customerGroups']));
					$customerGroup = $customer && $customer->getGroupCustomers() ? $customer->getGroupCustomers()->getId() : null;

					if (!in_array($customerGroup, $groups))
						continue;
				}

				$payment   = &$payments[$ps['payment']];
				$spedition = &$speditions[$ps['spedition']];

				if (!$payment || !$spedition || (Config::load('product.allowModifySpedition') && isset($disabledSpeditions[$ps['spedition']])))
					continue;

				$payment->vat = $spedition->vat;

				$dao           = new Dao\PaymentSpedition((int) $ps['id'], $payment, $spedition);
				$dao->currency = $ps['currency'] ?: null;

				if (EshopOrdersConfig::load('allowFreeFrom')) {
					$dao->freeFromDate = $ps['freeFromDate'];
					$dao->freeToDate   = $ps['freeToDate'];
				}

				foreach (explode(',', $ps['countries']) as $v) {
					$country = $this->countries->getDao()[$v];
					if ($country)
						$dao->addCountry($country->getId(), $country);
				}

				$this->cPublished[$ps['id']] = $dao;
			}

			if (EshopOrdersConfig::load('allowFreeFrom')) {
				foreach ($this->cPublished as $id => &$dao) {
					if ($dao->isFreeCombination()) {
						$dao->getSpedition()->freeForPayments[] = &$payments[$dao->getPayment()->getId()];
						$dao->getPayment()->freeForSpeditions[] = &$speditions[$dao->getSpedition()->getId()];
					}
				}
			}
		}

		return $this->cPublished;
	}

	public function getAllPublishedByCountry(?string $country = null): array
	{
		if (!$country)
			return $this->getAllPublished();

		$result = [];

		foreach ($this->getAllPublished() as $row) {
			if (isset($row->getCountries()[Strings::lower($country)]) || isset($row->getCountries()[Strings::upper($country)]))
				$result[$row->getId()] = $row;
		}

		return $result;
	}

	public function getFirstFreeSpedition(?string $country = null)
	{
		if (!EshopOrdersConfig::load('allowFreeFrom')) {
			return null;
		}

		/** @var Dao\Spedition|null $min */
		$min = null;

		foreach ($this->getAllPublished() as $ps) {
			if (!$country)
				continue;

			$countries = $ps->getCountries()[$country] ?? $ps->getCountries()[strtolower($country)] ?? [];
			if (empty($countries))
				continue;

			if ($ps->isFreeCombination()) {
				$sped           = clone $ps->getSpedition();
				$sped->freeFrom = 0;
				$min            = $sped;
				break;
			}

			$spedition = $ps->getSpedition();
			if ($spedition->freeFrom !== null && $spedition->freeFrom > 0 && ($min === null || $spedition->freeFrom < $min->freeFrom))
				$min = $spedition;
		}

		return $min;
	}

	public function getFirstFreeSpeditionByType(?string $country = null)
	{
		/** @var Dao\Spedition|null $minStorePickup */
		$minStorePickup = null;

		/** @var Dao\Spedition|null $minPickup */
		$minPickup = null;

		/** @var Dao\Spedition|null $minAddress */
		$minAddress = null;

		if (EshopOrdersConfig::load('allowFreeFrom')) {
			foreach ($this->getAllPublished() as $ps) {
				$countries = array_change_key_case($ps->getCountries(), CASE_UPPER);
				$country   = $country ? strtoupper($country) : $country;
				if (!$country || ($country && !empty($countries) && !isset($countries[$country])))
					continue;

				if ($ps->isFreeCombination()) {
					$sped           = clone $ps->getSpedition();
					$sped->freeFrom = 0;

					if ($sped->isPickup && $sped->getIdent() === 'pickup') {
						$minStorePickup = $sped;
					} else if ($sped->isPickup) {
						$minPickup = $sped;
					} else {
						$minAddress = $sped;
					}

					break;
				}
				$spedition = $ps->getSpedition();

				if ($spedition->freeFrom !== null && ($minStorePickup === null || $spedition->freeFrom < $minStorePickup->freeFrom) && $spedition->isPickup && $spedition->getIdent() === 'pickup') {
					$minStorePickup = $spedition;
				} else if ($spedition->freeFrom !== null && $spedition->freeFrom > 0 && ($minPickup === null || $spedition->freeFrom < $minPickup->freeFrom) && $spedition->isPickup) {
					$minPickup = $spedition;
				} else if ($spedition->freeFrom !== null && $spedition->freeFrom > 0 && ($minAddress === null || $spedition->freeFrom < $minAddress->freeFrom) && !$spedition->isPickup) {
					$minAddress = $spedition;
				}
			}
		}

		$result = [
			'storePickup' => $minStorePickup,
			'pickup'      => $minPickup,
			'address'     => $minAddress,
		];

		if (EshopOrdersConfig::load('allowFreeFrom')) {
			uasort($result, fn($sp1, $sp2) => $sp1->freeFrom <=> $sp2->freeFrom);
		}

		return $result;
	}

	public function remove($id)
	{
		$result = parent::remove($id);

		if ($result) {
			$this->eshopOrdersCache->getCache()->clean([
				Cache::Tags => ['payment', 'spedition', 'freeFrom'],
			]);
		}

		return $result;
	}

}
