<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Affiliate;

use Core\Model\Helpers\Strings;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopOrders\AdminModule\Model\AffiliatePartners;
use EshopOrders\FrontModule\Model\EshopOrdersCache;
use EshopOrders\Model\Entities\Affiliate\AffiliatePartner;
use Exception;
use InvalidArgumentException;
use Nette\Application\Attributes\Persistent;
use Nette\Utils\ArrayHash;

class PartnerForm extends BaseControl
{
	#[Persistent]
	public ?int $id = null;

	public ?AffiliatePartner $affiliatePartner = null;

	public function __construct(
		protected AffiliatePartners $affiliatePartners,
		protected EshopOrdersCache  $eshopOrdersCache,
	)
	{
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setShowLangSwitcher(false);
		$form->setAjax();

		$form->addText('ident', 'eshopOrders.affiliate.ident')
			->setDescription('eshopOrders.affiliate.identDescription');
		$form->addText('name', 'eshopOrders.affiliate.name')
			->setRequired();

		$form->addSaveCancelControl('saveControl');

		if ($this->id) {
			$form->getComponent('ident')->setDisabled();
		}

		$form->onValidate[] = $this->formValidate(...);
		$form->onSuccess[]  = $this->formSuccess(...);

		return $form;
	}

	public function formValidate(BaseForm $form, ArrayHash $values): void
	{
		if (!$this->id) {
			$ident  = Strings::webalize($values->ident ?: $values->name);
			$exists = $this->affiliatePartners->getEr()->findOneBy(['ident' => $ident]);

			if ($exists) {
				$form->getComponent('ident')->addError('eshopOrders.affiliate.identExists');
			}
		}

		if ($form->getErrors()) {
			$this->redrawControl('form');
		}
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			$ident = Strings::webalize($values->ident ?: $values->name);

			/** @var AffiliatePartner $partner */
			$partner = $this->id ? $this->affiliatePartners->get($this->id) : new AffiliatePartner($values->name, $ident);

			$partner->name = $values->name;

			$this->em->persist($partner);
			$this->em->flush();

			$this->eshopOrdersCache->cleanAffiliate();

			return true;
		} catch (Exception $e) {
			$form->addError($e->getMessage());
			$this->redrawControl('form');
		}

		return false;
	}

	public function setPartner(int $id): void
	{
		$this->id               = $id;
		$this->affiliatePartner = $this->affiliatePartners->get($id);

		if (!$this->affiliatePartner) {
			throw new InvalidArgumentException();
		}

		$d = [
			'ident' => $this->affiliatePartner->ident,
			'name'  => $this->affiliatePartner->name,
		];

		$this['form']->setDefaults($d);
	}
}
