<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Customer;

use Core\Model\Countries;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\Helpers\EntityHelpers;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopOrders\AdminModule\Model\Customers;
use EshopOrders\AdminModule\Model\Payments;
use EshopOrders\AdminModule\Model\Speditions;
use EshopOrders\Components\AddressForm;
use EshopOrders\Components\IAddressFormFactory;
use EshopOrders\Model\Entities\Customer;
use EshopOrders\Model\Entities\CustomerAddress;
use EshopOrders\Model\Entities\GroupCustomers;
use EshopOrders\Model\EshopOrdersConfig;
use Exception;
use Nette\Utils\ArrayHash;
use Users\Model\Users;
use Users\Model\UsersConfig;

class CustomerForm extends BaseControl
{
	public ?Customer $customer = null;

	protected ?array $cOtherUsersForLogin = null;

	public function __construct(
		protected Customers           $customersService,
		protected Countries           $countries,
		protected Payments            $payments,
		protected Speditions          $speditions,
		protected Users               $users,
		protected IAddressFormFactory $addressFormFactory,
	)
	{
	}

	public function render(): void
	{
		$users   = [];
		$allUser = $this->users->getAllNonAdminUsersBasicData();
		foreach ($this->getOtherUsersForLogin() as $id) {
			if (isset($allUser[$id])) {
				$users[$id] = $allUser[$id];
			}
		}

		if (EshopOrdersConfig::load('allowAddressManagement') && $this->customer) {
			$this->template->addresses    = $this->customer->addressesAvailable->toArray();
			$this->template->baseDelivery = $this->customer->getAddressDelivery() instanceof CustomerAddress ? $this->customer->getAddressDelivery()->getId() : null;
			$this->template->baseInvoice  = $this->customer->getAddressInvoice() instanceof CustomerAddress ? $this->customer->getAddressInvoice()->getId() : null;
		}

		$this->template->otherUsersForLogin = $users;
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleAddAddress(): void
	{
		$this['addressForm']->customerId = $this->customer instanceof Customer ? $this->customer->getId() : null;
		$this->template->modalTitle      = $this->t('eshopOrders.customerAddress.addForm');
		$this->template->modal           = 'addressForm';
		$this->redrawControl('modal');
	}

	public function handleEditAddress(int $id): void
	{
		$this['addressForm']->addressId  = $id;
		$this['addressForm']->customerId = $this->customer instanceof Customer ? $this->customer->getId() : null;
		$this->template->modalTitle      = $this->t('eshopOrders.customerAddress.editForm');
		$this->template->modal           = 'addressForm';
		$this->redrawControl('modal');
	}

	public function handleRemoveAddress(int $id): void
	{
		$address = $this->em->getRepository(CustomerAddress::class)->getReference($id);
		$this->em->remove($address);
		$this->em->flush();
		$this->presenter->flashMessageSuccess('eshopOrders.customerAddress.removed');
		$this->presenter->redrawControl('flashes');
		$this->redrawControl('addresses');
	}

	public function handleUpdateAddress(string $type, int $id): void
	{
		/** @var CustomerAddress $address */
		$address = $this->em->getReference(CustomerAddress::class, $id);
		if ($type === 'delivery') {
			$this->customer->setAddressDelivery($address);
		} else if ($type === 'invoice') {
			$this->customer->setAddressInvoice($address);
		}

		$this->em->persist($this->customer);
		$this->em->flush();
		$this->presenter->flashMessageSuccess('eshopOrders.customerAddress.updated');
		$this->presenter->redrawControl('flashes');
		$this->redrawControl('addresses');
	}

	public function handleLoadUsers(): void
	{
		$this->presenter->sendJson($this->users->getAllNonAdminUsersBasicData());
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();

		$groups = ['' => ''];
		foreach ($this->em->getRepository(GroupCustomers::class)->findPairs([], 'name', ['name'], 'id') as $k => $v) {
			$groups[$k] = $v;
		}

		$countries = $this->countries->getAllNameColumn();

		$form->addText('name', 'default.firstName')
			->setRequired()
			->setMaxLength(255);
		$form->addText('lastname', 'default.lastName')
			->setRequired()
			->setMaxLength(255);
		$form->addText('email', 'default.email')
			->setRequired()
			->setMaxLength(255);
		$form->addText('phone', 'default.phone')
			->setMaxLength(255);
		$form->addSelect('group', 'eshopOrders.customerForm.group', $groups);

		if (EshopOrdersConfig::load('enableDisablingPaymentSpeditionForCustomer')) {
			$form->addCheckboxList('disabledPayments', 'eshopOrders.customer.disabledPayments', $this->payments->getForSelectOption());
			$form->addCheckboxList('disabledSpeditions', 'eshopOrders.customer.disabledSpeditions', $this->speditions->getForSelectOption());
		}

		if (!EshopOrdersConfig::load('allowAddressManagement')) {
			$addressDelivery = $form->addContainer('addressDelivery');
			$addressInvoice  = $form->addContainer('addressInvoice');

			$fields = ['firstName', 'lastName', 'email', 'phone', 'company', 'street', 'city', 'postal', 'idNumber', 'vatNumber'];

			if (EshopOrdersConfig::load('customer.showIdVatNumber', false)) {
				$fields[] = 'idVatNumber';
			}

			$fields[] = 'country';

			foreach ($fields as $v) {
				foreach (['addressDelivery', 'addressInvoice'] as $com) {
					if ($v === 'email') {
						${$com}->addEmail($v, 'eshopOrders.default.' . $v)
							->setMaxLength(255);
					} else if ($v === 'country') {
						${$com}->addSelect($v, 'eshopOrders.default.' . $v, $countries);
					} else {
						${$com}->addText($v, 'eshopOrders.default.' . $v)
							->setMaxLength(255);
					}

					if ($v === 'idNumber' && EshopOrdersConfig::load('customer.allowValidateIdNumber')) {
						${$com}->addBool('validatedIdNumber', 'eshopOrders.default.validatedIdNumber');
					} else if ($v === 'vatNumber' && EshopOrdersConfig::load('customer.allowValidateVatNumber')) {
						${$com}->addBool('validatedVatNumber', 'eshopOrders.default.validatedVatNumber');
					}
				}
			}
		}

		if (EshopOrdersConfig::load('customer.allowMinimalOrderPrice', false)) {
			$form->addBool('allowMinimalOrderPrice', 'eshopOrders.customer.allowMinimalOrderPrice')
				->setDefaultValue(false);
			$form->addText('minimalOrderPrice', 'eshopOrders.customer.minimalOrderPrice')
				->setDefaultValue('0');
		}

		if (EshopOrdersConfig::load('customer.showBlacklisted', false)) {
			$form->addBool('isBlacklisted', 'eshopOrders.customer.isBlacklisted')
				->setDefaultValue(false);
		}

		if (EshopOrdersConfig::load('customer.allowInvoiceReminder', false)) {
			$invoiceReminderOptions = [];
			foreach (Customer::$invoiceReminderOptions as $v) {
				$invoiceReminderOptions[$v] = $this->t('eshopOrders.customer.invoiceReminder.items.' . $v);
			}

			$form->addCheckboxList('invoiceReminder', 'eshopOrders.customer.invoiceReminder.caption', $invoiceReminderOptions);
		}

		if (EshopOrdersConfig::load('customer.allowShowPricesWithoutVat', false)) {
			$form->addBool('showPricesWithoutVat', 'eshopOrders.customer.showPricesWithoutVat')
				->setDefaultValue(0);
		}

		$form->addBool('isActive', 'default.isActive')->setRequired();
		$form->addSaveCancelControl('saveControl');

		$form->onSuccess[] = $this->formSuccess(...);

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			if (!$this->customer instanceof Customer) {
				$this->presenter->flashMessageWarning('eshopOrders.customerForm.noCreate');

				return false;
			}

			$customer     = $this->customer;
			$user         = $customer->getUser();
			$flashMessage = 'eshopOrders.customerForm.edited';

			if (EshopOrdersConfig::load('customer.showBlacklisted', false)) {
				$customer->isBlacklisted = (int) $values->isBlacklisted;
			}

			if (EshopOrdersConfig::load('customer.allowInvoiceReminder', false)) {
				/** @phpstan-ignore-next-line */
				$customer->invoiceReminderCount = (int) max(array_map('intval', $values->invoiceReminder));
			}

			$customer->setPhone($values->phone);
			$user->setName($values->name);
			$user->setLastname($values->lastname);
			$user->setEmail($values->email);
			if ($values->group) {
				/** @var GroupCustomers $groupRef */
				$groupRef = $this->em->getReference(GroupCustomers::class, $values->group);

				$customer->setGroupCustomers($groupRef);
			} else {
				$customer->removeGroupCustomers();
			}

			if ($this->customersService->isAdmin($customer->getId()) && $values->isActive != $user->isActive()) {
				$this->presenter->flashMessageWarning('eshopOrders.customerForm.noAdminDeactivate');
			} else {
				$user->disable(!$values->isActive);
			}

			if (!EshopOrdersConfig::load('allowAddressManagement')) {
				$addressDelivery = $customer->getAddressDelivery() ?: new CustomerAddress($customer);
				$addressInvoince = $customer->getAddressInvoice() ?: new CustomerAddress($customer);

				$customer->setAddressDelivery($addressDelivery);
				$customer->setAddressInvoice($addressInvoince);

				$fields = ['firstName', 'lastName', 'email', 'phone', 'company', 'street', 'city', 'postal', 'idNumber', 'vatNumber'];
				if (EshopOrdersConfig::load('customer.showIdVatNumber', false)) {
					$fields[] = 'idVatNumber';
				}
				$fields[] = 'country';

				foreach ($fields as $v) {
					foreach (['addressDelivery', 'addressInvoice'] as $com) {
						if ($v === 'country') {
							if ($values->$com->$v) {
								$customer->$com->$v = $this->countries->getReference($values->$com->$v);
							}
						} else {
							$customer->$com->$v = $values->$com->$v;
						}

						if ($v === 'idNumber' && EshopOrdersConfig::load('customer.allowValidateIdNumber')) {
							$customer->$com->validatedIdNumber = (int) $values->$com->validatedIdNumber;
						} else if ($v === 'vatNumber' && EshopOrdersConfig::load('customer.allowValidateVatNumber')) {
							$customer->$com->validatedVatNumber = (int) $values->$com->validatedVatNumber;
						}
					}
				}

				$this->em->persist($addressDelivery);
				$this->em->persist($addressInvoince);
			}

			if (EshopOrdersConfig::load('enableDisablingPaymentSpeditionForCustomer')) {
				$currentDisabledPayments = $customer->disabledPayments->toArray();
				foreach ($values->disabledPayments as $id) {
					if (!isset($currentDisabledPayments[$id])) {
						$customer->disabledPayments->set($id, $this->payments->getReference($id));
					}

					unset($currentDisabledPayments[$id]);
				}

				foreach ($currentDisabledPayments as $entity) {
					$customer->disabledPayments->remove($entity->getId());
				}

				$currentDisabledSpeditions = $customer->disabledSpeditions->toArray();
				foreach ($values->disabledSpeditions as $id) {
					if (!isset($currentDisabledSpeditions[$id])) {
						$customer->disabledSpeditions->set($id, $this->speditions->getReference($id));
					}

					unset($currentDisabledSpeditions[$id]);
				}

				foreach ($currentDisabledSpeditions as $entity) {
					$customer->disabledSpeditions->remove($entity->getId());
				}
			}

			$this->em->persist($customer);
			$this->em->persist($user);

			$httpData = (array) $form->getHttpData();

			if (UsersConfig::load('allowLoginAsAnotherUser')) {
				$otherLogins        = $httpData['customerOtherLogins'] ?? [];
				$currentOtherLogins = $this->getOtherUsersForLogin();

				// set
				foreach (array_diff($otherLogins, $currentOtherLogins) as $id) {
					$ref          = $this->users->getReference($id);
					$ref->loginAs = $user;

					$this->em->persist($ref);
				}

				// remove
				foreach (array_diff($currentOtherLogins, $otherLogins) as $id) {
					$ref          = $this->users->getReference($id);
					$ref->loginAs = null;

					$this->em->persist($ref);
				}
			}

			$this->em->flush();

			if (EshopOrdersConfig::load('customer.allowShowPricesWithoutVat', false)) {
				$efShowPricesWithoutVat = EntityHelpers::setExtraField($customer, 'showPricesWithoutVat', (string) $values->showPricesWithoutVat);
				$this->em->persist($efShowPricesWithoutVat);
				$this->em->flush();
			}

			if (EshopOrdersConfig::load('customer.allowMinimalOrderPrice', false)) {
				$efAllowMinimalOrderPrice = EntityHelpers::setExtraField($customer, 'allowMinimalOrderPrice', (string) $values->allowMinimalOrderPrice);
				$efMinimalOrderPrice      = EntityHelpers::setExtraField($customer, 'minimalOrderPrice', (string) $values->minimalOrderPrice);

				$this->em->persist($efAllowMinimalOrderPrice);
				$this->em->persist($efMinimalOrderPrice);
				$this->em->flush();
			}

			$event                           = new FormSuccessEvent($form, $values, $this->template, $this->presenter);
			$event->custom['entity']         = $user;
			$event->custom['entityCustomer'] = $customer;
			$this->eventDispatcher->dispatch($event, self::class . '::formSuccessAfter');

			$form->addCustomData('customerId', $customer->getId());
			$this->presenter->flashMessageSuccess($flashMessage);
		} catch (Exception $e) {
			$form->addError($e->getMessage());
			$this->redrawControl('form');

			return false;
		}

		return true;
	}

	public function setCustomer(int $id): void
	{
		$this->customer = $customer = $this->customersService->get($id);

		if ($customer) {
			$o = $customer;
			$u = $o->getUser();

			$d = [
				'phone'         => $o->getPhone(),
				'name'          => $u->getName(),
				'lastname'      => $u->getLastname(),
				'email'         => $u->getEmail(),
				'isActive'      => $u->isActive(),
				'isBlacklisted' => $o->isBlacklisted,
			];

			for ($i = 0; $i <= $customer->invoiceReminderCount; $i++) {
				$d['invoiceReminder'][] = $i;
			}

			if (EshopOrdersConfig::load('enableDisablingPaymentSpeditionForCustomer')) {
				$d['disabledPayments']   = $o->disabledPayments->getKeys();
				$d['disabledSpeditions'] = $o->disabledSpeditions->getKeys();
			}

			if (!EshopOrdersConfig::load('allowAddressManagement')) {
				$fields = ['firstName', 'lastName', 'email', 'phone', 'company', 'street', 'city', 'postal', 'idNumber', 'vatNumber'];
				if (EshopOrdersConfig::load('customer.showIdVatNumber', false)) {
					$fields[] = 'idVatNumber';
				}
				$fields[] = 'country';

				foreach ($fields as $v) {
					foreach (['addressDelivery', 'addressInvoice'] as $com) {
						if (!$customer->$com) {
							continue;
						}

						if ($v === 'country') {
							/** @phpstan-ignore-next-line */
							$d[$com][$v] = $customer->$com->$v ? $customer->$com->$v->getId() : null;
						} else {
							/** @phpstan-ignore-next-line */
							$d[$com][$v] = $customer->$com->$v;
						}

						if ($v === 'idNumber' && EshopOrdersConfig::load('customer.allowValidateIdNumber')) {
							$d[$com]['validatedIdNumber'] = $customer->$com->validatedIdNumber;
						} else if ($v === 'vatNumber' && EshopOrdersConfig::load('customer.allowValidateVatNumber')) {
							$d[$com]['validatedVatNumber'] = $customer->$com->validatedVatNumber;
						}
					}
				}
			}

			$ef = $customer->getExtraFieldsValues();
			if (EshopOrdersConfig::load('customer.allowMinimalOrderPrice', false)) {
				foreach (['allowMinimalOrderPrice', 'minimalOrderPrice'] as $item) {
					$d[$item] = $ef[$item];
				}
			}

			if (EshopOrdersConfig::load('customer.allowShowPricesWithoutVat', false)) {
				$d['showPricesWithoutVat'] = $ef['showPricesWithoutVat'];
			}

			$this['form']->setDefaults($d);

			if ($o->getGroupCustomers() && array_key_exists($o->getGroupCustomers()->getId(), $this['form']->getComponent('group')->getItems())) {
				$this['form']->getComponent('group')->setDefaultValue($o->getGroupCustomers()->getId());
			}
		} else {
			$this->presenter->error('eshopOrders.customer.notFound');
		}
	}

	protected function getOtherUsersForLogin(): array
	{
		if (!$this->customer instanceof Customer) {
			return [];
		}

		if ($this->cOtherUsersForLogin === null) {
			$this->cOtherUsersForLogin = $this->customersService->getAllOtherUsersForLogin($this->customer->getUser()->getId());
		}

		return $this->cOtherUsersForLogin;
	}

	protected function createComponentAddressForm(): AddressForm
	{
		$control = $this->addressFormFactory->create();
		if ($this->getParameter('id')) {
			$control->setAddress((int) $this->getParameter('id'));
		}

		$control['form']->onSuccessSave[]         = function(BaseForm $form): void {
			$this->presenter->redrawControl('flashes');
			$this->redrawControl('addresses');
		};
		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form): void {
			$this->presenter->payload->hideModal = true;
			$this->presenter->redrawControl('flashes');
			$this->redrawControl('addresses');
		};
		$control['form']->getComponent('saveControl')->closeModalOnCancel();

		return $control;
	}
}
