<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Customer;

use Core\Model\Helpers\Strings;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Doctrine\ORM\QueryBuilder;
use EshopOrders\AdminModule\Model\Customers;
use EshopOrders\Model\Entities\Customer;
use EshopOrders\Model\Entities\CustomerAddress;
use EshopOrders\Model\Entities\GroupCustomers;
use EshopOrders\Model\EshopOrdersConfig;

class CustomersGrid extends BaseControl
{
	public function __construct(
		protected Customers $customersService,
	)
	{
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();
		$grid->setStrictSessionFilterValues(false);

		$qb = $this->customersService->getEr()->createQueryBuilder('p')
			->addSelect('user, adDel, adInv, grp')
			->innerJoin('p.user', 'user')
			->leftJoin('p.addressDelivery', 'adDel')
			->leftJoin('p.addressInvoice', 'adInv')
			->leftJoin('p.groupCustomers', 'grp')
			->orderBy('p.id');
		$grid->setDataSource($qb);

		$groupCustomers = [];
		foreach ($this->em->getRepository(GroupCustomers::class)->createQueryBuilder('gc')
			         ->select('gc.id, gc.name')->getQuery()->getArrayResult() as $gc) {
			$groupCustomers[$gc['id']] = $gc['name'];
		}

		//Columns
		if (EshopOrdersConfig::load('customersGrid.showCompany')) {
			$grid->addColumnText('company', 'eshopOrders.default.company')
				->setRenderer(fn(Customer $row): ?string => $row->getAddressInvoice() instanceof CustomerAddress ? $row->getAddressInvoice()->getCompany() : '')
				->setFilterText()
				->setCondition(function(QueryBuilder $qb, $value): void {
					$qb->andWhere('adInv.company LIKE :company OR adDel.company LIKE :company')
						->setParameter('company', "%$value%");
				});
		}
		$grid->addColumnLink('name', 'default.firstName', 'Customers:editCustomer', 'user.name');
		$grid->addColumnLink('lastname', 'default.lastName', 'Customers:editCustomer', 'user.lastname');
		$grid->addColumnEmail('email', 'default.email', 'user.email');
		$grid->addColumnPhone('phone', 'default.phone', 'phone', 'addressInvoice.country.id', true);

		if (EshopOrdersConfig::load('customersGrid.showIco', false)) {
			$grid->addColumnText('idNumber', 'eshopOrders.default.idNumber', 'addressInvoice.idNumber')
				->setFitContent()
				->setFilterText()->setCondition(function(QueryBuilder $qb, $value): void {
					$qb->andWhere('adInv.idNumber LIKE :idNumber')
						->setParameter('idNumber', "%$value%");
				});
		}

		$grid->addColumnText('group', 'default.group')->setRenderer(fn(Customer $row): string => $row->getGroupCustomers() instanceof GroupCustomers ? $row->getGroupCustomers()->name : '');
		$grid->addColumnStatus('isActive', 'default.isActive', 'user.isActive')->setAlign('center')
			->addOption(1, 'eshopOrders.defaultGrid.publish')->setIcon('check')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(0, 'eshopOrders.defaultGrid.unPublish')->setIcon('times')->setClass('btn-danger')->setShowTitle(false)->endOption()
			->onChange[] = $this->gridPublishChange(...);

		// Filter
		$grid->addFilterText('name', '', 'user.name');
		$grid->addFilterText('lastname', '', 'user.lastname');
		$grid->addFilterText('email', '', 'user.email');
		$grid->addFilterText('phone', '', 'adDel.phone');
		$grid->addFilterSelect('group', '', [
				null => '',
				'no' => $this->t('eshopOrders.customer.withoutGroup'),
			] + $groupCustomers, 'grp.id')
			->setCondition(function(QueryBuilder $qb, $value): void {
				if ($value === 'no') {
					$qb->andWhere('grp.id IS NULL');
				} else {
					$qb->andWhere('grp.id = :customerGroup')->setParameter('customerGroup', $value);
				}
			});
		$grid->addFilterSelect('isActive', '', [
			null => '',
			1    => $this->t('default.yes'),
			0    => $this->t('default.no'),
		], 'user.isActive');

		// Actions
		$grid->addAction('edit', '', 'Customers:editCustomer')->setIcon('edit')->setBsType('primary');
		$grid->addGroupSelectAction('eshopOrders.defaultGrid.editGroupCustomers',
			[0 => $this->t('eshopOrders.defaultGrid.nullGroupCustomers')] + $groupCustomers)
			->onSelect[] = $this->gridEditGroupCustomers(...);

		// Export
		if (EshopOrdersConfig::load('customersGrid.allowExport')) {
			$grid->addExportCsvFiltered('eshopOrders.customersGrid.filteredExportCsv', 'customersGrid')
				->setColumns([
					$grid->getColumn('name'),
					$grid->getColumn('lastname'),
					$grid->getColumn('email'),
					$grid->getColumn('phone')->setRenderer(function(Customer $customer) {
						$phone = Strings::phoneFormat($customer->getPhone(), null, false);
						if (!\str_starts_with($phone, '+')) {
							$phone = '+' . $phone;
						}

						return $phone;
					}),
					$grid->getColumn('group'),
					$grid->getColumn('isActive'),
				]);
		}

		return $grid;
	}


	/*******************************************************************************************************************
	 * ==================  Handle
	 */


	/*******************************************************************************************************************
	 * =================  Grid function
	 */

	public function gridPublishChange(string $id, string $newStatus): void
	{
		if ($this->customersService->isAdmin((int) $id)) {
			$this->presenter->flashMessageWarning('eshopOrders.customerForm.noAdminDeactivate');
		} else if ($this->customersService->setPublish((int) $id, (int) $newStatus)) {
			$this->presenter->flashMessageSuccess('eshopOrders.defaultGrid.publishChanged');
		} else {
			$this->presenter->flashMessageDanger('eshopOrders.defaultGrid.publishChangeFailed');
		}

		$this['grid']->redrawItem($id);
		$this->presenter->redrawControl('flashes');
	}

	public function gridEditGroupCustomers(array $ids, int $groupCustomers): void
	{
		if ($this->customersService->setGroupCustomers($ids, $groupCustomers)) {
			$this->presenter->flashMessageSuccess('eshopOrders.defaultGrid.groupCustomersChanged');
		} else {
			$this->presenter->flashMessageDanger('eshopOrders.defaultGrid.groupCustomersChangeFailed');
		}

		$this['grid']->reload();
		$this->presenter->redrawControl('flashes');
	}

}
