<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Invoice;

use Core\Model\Entities\QueryBuilder;
use Core\Model\Templating\Filters\Price;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\DataGrid\DataSource\DoctrineDataSource;
use Core\Model\UI\Form\BaseForm;
use EshopOrders\AdminModule\Components\Order\IIsPaidSwitchFactory;
use EshopOrders\AdminModule\Components\Order\IsPaidSwitch;
use EshopOrders\Model\Entities\Invoice;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Invoices;
use Nette\Application\UI\Multiplier;
use Nette\Utils\ArrayHash;
use Nette\Utils\Html;
use Nette\Utils\Validators;

class InvoicesGrid extends BaseControl
{
	public function __construct(
		protected Invoices             $invoices,
		protected Price                $priceFilter,
		protected IIsPaidSwitchFactory $isPaidSwitchFactory,
	)
	{
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	public function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		if (EshopOrdersConfig::load('invoicesGrid.setRefreshUrl')) {
			$grid->setRefreshUrl(true);
		}

		$qb = $this->invoices->getQueryBuilder()
			->addSelect('iData, pay, sped, prods, disco, cust, o, oCurr')
			->leftJoin('i.invoiceData', 'iData')
			->leftJoin('iData.payment', 'pay')
			->leftJoin('iData.spedition', 'sped')
			->leftJoin('iData.customer', 'cust')
			->leftJoin('iData.products', 'prods')
			->leftJoin('iData.discounts', 'disco')
			->leftJoin('i.order', 'o')
			->leftJoin('o.currency', 'oCurr')
			->orderBy('i.createdAt', 'desc');

		$dataSource = new DoctrineDataSource($qb, 'i.id');

		$dataSource->onDataLoaded[] = function($items) {
			foreach ($items as $item) {
				$this->invoices->setPriceCalculator($item);
			}
		};

		$grid->setDataSource($dataSource);

		// columns
		$grid->addColumnText('ident', 'eshopOrders.invoicesGrid.ident')
			->setFilterText();
		$grid->addColumnLink('order', 'eshopOrders.invoicesGrid.order', ':EshopOrders:Admin:Default:editOrder', 'order.id')
			->setRenderer(fn(Invoice $invoice) => $invoice->order instanceof Order
				? Html::el('a', [
					'href' => $this->getPresenter()->link(':EshopOrders:Admin:Default:editOrder', ['id' => $invoice->order->getId()]),
				])->setText($invoice->order->getId())
				: '')
			->setFilterText()
			->setCondition(function(QueryBuilder $qb, $val): void {
				$qb->andWhere('o.id LIKE :order')
					->setParameter('order', "%{$val}%");
			});
		$grid->addColumnText('subscriber', 'eshopOrders.invoicesGrid.subscriber')->setRenderer(static function(Invoice $invoice) {
			if ($invoice->isExternal()) {
				return '';
			}

			$customer = $invoice->invoiceData->getCustomer();

			return Validators::isNone($customer->company) ? sprintf('%s %s', $customer->firstName, $customer->lastName) : $customer->company;
		})
			->setFilterText()
			->setCondition(function(QueryBuilder $qb, $val): void {
				$qb->andWhere('cust.company LIKE :customer OR cust.firstName LIKE :customer OR cust.lastName LIKE :customer')
					->setParameter('customer', "%{$val}%");
			});
		$grid->addColumnText('payment', 'eshopOrders.invoicesGrid.payment', 'invoiceData.payment.name');
		$grid->addColumnDateTime('paid', 'eshopOrders.invoicesGrid.paid', 'order.paid');
		$grid->addColumnDateTime('fromDate', 'eshopOrders.invoicesGrid.fromDate')->setRenderer(static fn(Invoice $invoice) => $invoice->createdAt->format('d. m. Y'))
			->setFilterDate('createdAt');
		$grid->addColumnDateTime('dueDate', 'eshopOrders.invoicesGrid.dueDate')
			->setFilterDate('createdAt');
		$grid->addColumnText('total', 'eshopOrders.invoicesGrid.totalPrice')->setRenderer(function(Invoice $invoice) {
			if ($invoice->isExternal()) {
				return '';
			}

			if ($invoice->order instanceof Order) {
				$invoice = $this->invoices->getOneByOrder($invoice->order);

				return $invoice->order->renderPrice($invoice->invoiceData->getPrice(), true);
			}

			return $this->priceFilter->format($invoice->invoiceData->getPrice());
		});

		if (EshopOrdersConfig::load('invoicesGrid.showIsPaidSwitch')) {
			$grid->addColumnText('isPaid', 'eshopOrders.invoicesGrid.isPaid')->setRenderer(function(Invoice $invoice): void {
				if (!$invoice->order instanceof Order) {
					return;
				}

				$this['isPaidSwitch-' . $invoice->order->getId()]->render();
			})->setAlign('center');
			$grid->getColumn('isPaid')->setFilterSelect([
				''  => '',
				'0' => $this->t('default.no'),
				'1' => $this->t('default.yes'),
			])->setCondition(function(QueryBuilder $qb, $value): void {
				$qb->andWhere('o.isPaid = :oIsPaid')
					->setParameter('oIsPaid', $value);
			});
			$grid->getColumn('isPaid')->getElementPrototype('td')->addClass('w1nw');
		}

		// actions
		$grid->addAction('pdf', '')->setRenderer(function(Invoice $invoice) {
			if (!$invoice->order instanceof Order) {
				return '';
			}

			return Html::el('a', ['href'  => $this->presenter->link('Orders:invoice', $invoice->order->getId()),
			                      'class' => 'btn btn-danger btn-xs', 'target' => '_blank'])
				->addHtml(Html::el('i', ['class' => 'fas fa-file-invoice']))
				->addHtml(Html::el('span')->setText($this->t('eshopOrders.invoicesGrid.pdf')));
		});

		if ($this->getPresenter()->getUser()->isAllowed('EshopOrders:Admin:Invoice', 'regenerateInvoices')) {
			$grid->addAction('regenerate', 'eshopOrders.invoicesGrid.actions.regenerate', 'regenerate!')
				->setRenderCondition(fn(Invoice $invoice): bool => !$invoice->isExternal())
				->setRenderer(function(Invoice $invoice) {
					if (!$invoice->order instanceof Order) {
						return '';
					}

					return Html::el('a', ['href'  => $this->presenter->link('regenerate!', $invoice->getId()),
					                      'class' => 'btn btn-danger btn-xs ajax'])
						->addHtml(Html::el('i', ['class' => 'fas fa-sync']))
						->addHtml(Html::el('span')->setText($this->t('eshopOrders.invoicesGrid.actions.regenerate')));
				})
				->setConfirm('eshopOrders.invoice.reallyRegenerate');
		}

		return $grid;
	}

	protected function createComponentIsPaidSwitch(): Multiplier
	{
		return new Multiplier(function(string $id): IsPaidSwitch {
			/** @var Order $order */
			$order = $this->em->getReference(Order::class, $id);

			$control = $this->isPaidSwitchFactory->create($order);

			$control['form']->onSuccess[] = function(BaseForm $form, ArrayHash $values): void {
				$this->presenter->flashMessageSuccess('default.saved');
				$this->presenter->redrawControl('flashes');
			};

			return $control;
		});
	}
}
