<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Invoice;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use DateTime;
use DateTimeImmutable;
use EshopOrders\Model\Entities\Invoice;
use EshopOrders\Model\Entities\InvoiceConfig;
use EshopOrders\Model\Entities\NumericalSeries;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\InvoiceConfigRepository;
use Exception;
use Nette\ComponentModel\IComponent;
use Tracy\Debugger;

class QuickEditForm extends BaseControl
{
	public function __construct(
		protected ?Invoice                $invoice,
		protected InvoiceConfigRepository $invoiceConfigRepository,
	)
	{
	}

	public function render(): void
	{
		if (!$this->invoice instanceof Invoice) {
			return;
		}

		$this->template->render($this->getTemplateFile());
	}

	public function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();
		$form->setShowLangSwitcher(false);

		if (EshopOrdersConfig::load('orderForm.allowEditInvoiceDUZP', false)) {
			$duzpControl = $form->addDatePicker('duzp', 'eshopOrders.orderDetail.invoiceDUZP')->setDefaultValue(new DateTime);

			if (EshopOrdersConfig::load('orderForm.allowValidateInvoiceDUZP', false) && $this->invoice) {
				$numericalSeries = $this->getNumericalSeries();
				if ($numericalSeries && $numericalSeries->containsPrefixYearWildcards()) {
					$duzpControl->setHtmlAttribute('data-validate-year', 'true');
				}

				if ($numericalSeries && $numericalSeries->containsPrefixMonthWildcards()) {
					$duzpControl->setHtmlAttribute('data-validate-month', 'true');
				}
			}
		}

		if (EshopOrdersConfig::load('orderForm.allowInvoiceCustomerCustomNumber', false)) {
			$form->addText('customerCustomNumber', 'eshopOrders.orderDetail.customerCustomNumber');
		}

		if (EshopOrdersConfig::load('orderForm.allowEditInvoiceDueDate', false) && $this->invoice) {
			$form->addDatePicker('dueDate', 'eshopOrders.orderDetail.invoiceDueDate')
				->setNullable();
		}

		$form->addButton('save', $this->t('default.save'));

		if ($this->invoice instanceof Invoice) {
			if ($form->getComponent('duzp', false) instanceof IComponent) {
				$form->getComponent('duzp')->setDefaultValue($this->invoice->getDUZP());
			}

			if ($form->getComponent('dueDate', false) instanceof IComponent) {
				$form->getComponent('dueDate')->setDefaultValue($this->invoice->dueDate);
			}

			if ($form->getComponent('customerCustomNumber', false) && $this->invoice->invoiceData->customer) {
				$form->getComponent('customerCustomNumber')->setDefaultValue($this->invoice->invoiceData->customer->customNumber);
			}
		}

		$form->addHidden('defaultDUZPYear');
		$form->addHidden('defaultDUZPMonth');
		if ($form->getComponent('duzp', false) instanceof IComponent) {
			/** @var DateTimeImmutable $duzpDate */
			$duzpDate = $form->getComponent('duzp')->getValue();
			$form->getComponent('defaultDUZPYear')->setValue($duzpDate->format('Y'));
			$form->getComponent('defaultDUZPMonth')->setValue($duzpDate->format('n'));
		}

		$form->onSuccess[] = function(BaseForm $form, array $values): void {
			if (!$this->invoice instanceof Invoice) {
				return;
			}

			try {
				$this->em->beginTransaction();

				if (array_key_exists('dueDate', $values) && $values['dueDate']) {
					$this->invoice->dueDate = DateTime::createFromImmutable($values['dueDate']);
				}

				$oldDuzp             = clone $this->invoice->getDUZP();
				$oldDuzpYear         = (int) $oldDuzp->format('Y');
				$oldDuzpMonth        = (int) $oldDuzp->format('n');
				$this->invoice->duzp = $values['duzp'] ? DateTime::createFromImmutable($values['duzp']) : null;
				$newDuzp             = clone $this->invoice->duzp;
				$newDuzpYear         = (int) $newDuzp->format('Y');
				$newDuzpMonth        = (int) $newDuzp->format('n');

				if ($this->invoice->invoiceData->customer) {
					$this->invoice->invoiceData->customer->customNumber = $values['customerCustomNumber'];
					$this->em->persist($this->invoice->invoiceData->customer);
				}

				// pri zmene duzp roku ci mesice, se vygeneruje novy ident faktury
				if ($this->invoice->order && ($numSeries = $this->getNumericalSeries()) && (($numSeries->containsPrefixYearWildcards() && $oldDuzpYear !== $newDuzpYear) ||
						($numSeries->containsPrefixMonthWildcards() && $oldDuzpMonth !== $newDuzpMonth))) {
					$type = InvoiceConfig::TYPE_INVOICE;
					if ($this->invoice->order && $this->invoice->order->isCorrectiveTaxDocument) {
						$type = InvoiceConfig::TYPE_CORRECTIVE;
					}
					$this->invoice->ident = (string) $this->invoiceConfigRepository->generateIdent($this->invoice->order, (int) $this->invoice->seller->getId(), $type);
				}

				$form->getComponent('defaultDUZPYear')->setValue($newDuzpYear);
				$form->getComponent('defaultDUZPMonth')->setValue($newDuzpMonth);

				$this->em->persist($this->invoice);
				$this->em->flush();

				$this->presenter->flashMessageSuccess('default.saved');
				$this->presenter->redrawControl('flashes');

				$this->em->commit();

				$this->redrawControl('form');
			} catch (Exception $ex) {
				Debugger::log($ex);
				if ($this->em->getConnection()->isTransactionActive()) {
					$this->em->getConnection()->rollBack();
				}
			}
		};

		return $form;
	}

	protected function getNumericalSeries(): ?NumericalSeries
	{
		if (!$this->invoice instanceof Invoice) {
			return null;
		}

		$type = InvoiceConfig::TYPE_INVOICE;
		if ($this->invoice->order && $this->invoice->order->isCorrectiveTaxDocument) {
			$type = InvoiceConfig::TYPE_CORRECTIVE;
		}

		return $this->invoiceConfigRepository->getNumericalSeries((int) $this->invoice->seller->getId(), $type);
	}

}
