<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Order;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopOrders\AdminModule\Model\Orders;
use EshopOrders\AdminModule\Model\Payments;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderDiscount;
use EshopOrders\Model\Entities\OrderItem;
use EshopOrders\Model\EshopOrdersConfig;
use Exception;
use Nette\Utils\ArrayHash;
use Tracy\Debugger;

class CorrectiveTaxDocumentForm extends BaseControl
{
	public function __construct(
		protected Order    $order,
		protected Orders   $orders,
		protected Payments $payments,
	)
	{
	}

	public function render(): void
	{
		$this->template->order = $this->order;
		$this->template->render($this->getTemplateFile());
	}

	public function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax()
			->setShowLangSwitcher(false);

		/** @var OrderItem $oi */
		foreach ($this->order->getOrderItems()->toArray() as $oi) {
			$container = $form->addContainer('prod_' . $oi->getId());
			$container->addHidden('oiId', $oi->getId());
			$price = $oi->order->isZeroVat() ? $oi->getPriceRaw(true) : $oi->getPrice();
			$container->addText('price')
				->setDefaultValue($price)
				->addCondition($form::FILLED)
				->addRule($form::FLOAT)
				->addRule($form::MIN, null, 0)
				->addRule($form::MAX, null, $price);
			$container->addInteger('quantity')
				->setDefaultValue($oi->getQuantity())
				->addCondition($form::FILLED)
				->addRule($form::MIN, null, 1)
				->addRule($form::MAX, null, $oi->getQuantity());
		}

		$orderDiscounts = $this->order->getOrderDiscounts()->toArray();
		usort($orderDiscounts, fn(OrderDiscount $o1, OrderDiscount $o2): int => $o1->getSortPosition() <=> $o2->getSortPosition());
		/** @var OrderDiscount $od */
		foreach ($orderDiscounts as $od) {
			$container = $form->addContainer('discount_' . $od->getId());
			$container->addHidden('odId', $od->getId());
			$container->addText('price')
				->setDefaultValue(abs($od->getPrice()))
				->addCondition($form::FILLED)
				->addRule($form::FLOAT)
				->addRule($form::MIN, null, 0);
		}

		if (EshopOrdersConfig::load('orderForm.allowEditInvoiceDUZP', false) && $this->order->getInvoice()) {
			$form->addDatePicker('duzp', 'eshopOrders.correctiveTaxDocumentForm.invoiceDUZP')
				->setDefaultValue($this->order->getInvoice()->getDUZP());

			$form->addDatePicker('createdAt', 'eshopOrders.correctiveTaxDocumentForm.createdAt')
				->setDefaultValue($this->order->getInvoice()->createdAt);
		}

		$form->addSelect('payment', 'eshopOrders.correctiveTaxDocumentForm.payment', $this->payments->getForSelectOption())
			->setRequired();

		$defaultPaymentIdent = $paymentIdent = EshopOrdersConfig::load('correctiveTaxDocument.payment');
		if ($defaultPaymentIdent) {
			$defaultPaymentId = $this->em->getConnection()->fetchOne("SELECT id FROM eshop_orders__payment WHERE ident = :ident LIMIT 1", [
				'ident' => $defaultPaymentIdent,
			]);

			if ($defaultPaymentId && array_key_exists($defaultPaymentId, $form->getComponent('payment')->getItems())) {
				$form->getComponent('payment')->setDefaultValue($defaultPaymentId);
			}
		}

		$paymentPrice = isset($oi) && $oi->order->isZeroVat() ? $this->order->getPayment()->getPriceRaw() : $this->order->getPayment()->getPrice();
		$form->addText('paymentPrice', 'eshopOrders.correctiveTaxDocumentForm.paymentPrice')
			->setDefaultValue($paymentPrice)
			->setRequired()
			->addRule($form::FLOAT)
			->addRule($form::MIN, null, 0);
		$speditionPrice = isset($oi) && $oi->order->isZeroVat() ? $this->order->getSpedition()->getPriceRaw() : $this->order->getSpedition()->getPrice();
		$form->addText('speditionPrice', 'eshopOrders.correctiveTaxDocumentForm.speditionPrice')
			->setDefaultValue($speditionPrice)
			->setRequired()
			->addRule($form::FLOAT)
			->addRule($form::MIN, null, 0);

		$form->addSaveCancelControl();

		$form->onValidate[] = function(BaseForm $form, ArrayHash $values): void {
			if ($form->hasErrors()) {
				$this->redrawControl('formErrors');
			}
		};
		$form->onSuccess[]  = $this->formSuccess(...);

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): void
	{
		$this->em->beginTransaction();

		try {
			$products = [];
			foreach ((array) $form->getHttpData() as $k => $container) {
				if (!\str_starts_with((string) $k, 'prod_')) {
					continue;
				}
				$productId            = (int) explode('_', (string) $k)[1];
				$products[$productId] = $container;
			}
			$orderDiscounts = [];
			foreach ((array) $form->getHttpData() as $k => $container) {
				if (!\str_starts_with((string) $k, 'discount_')) {
					continue;
				}
				$discountId                  = (int) explode('_', (string) $k)[1];
				$orderDiscounts[$discountId] = $container;
			}

			$vals = (array) $values;
			$this->orders->createCorrectiveTaxDocument(
				$this->order,
				$products,
				$orderDiscounts,
				$values->paymentPrice,
				$values->speditionPrice,
				array_key_exists('duzp', $vals) ? $values->duzp : null,
				array_key_exists('createdAt', $vals) ? $values->createdAt : null,
				(int) $values->payment
			);

			$this->em->commit();

			$this->presenter->flashMessageSuccess('eshopOrders.correctiveTaxDocumentForm.created');
		} catch (Exception $e) {
			Debugger::log($e);
			if ($this->em->getConnection()->isTransactionActive()) {
				$this->em->rollback();
			}
			$this->presenter->flashMessageDanger('eshopOrders.correctiveTaxDocumentForm.savedError');
		}
	}
}
