<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Order;

use Core\Model\Countries;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderAddress;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Orders;
use Exception;
use InvalidArgumentException;
use Nette\Application\Attributes\Persistent;
use Nette\ComponentModel\IComponent;
use Nette\Utils\ArrayHash;
use Override;

class OrderAddressForm extends BaseControl
{
	#[Persistent]
	public ?int $orderId = null;

	#[Persistent]
	public ?string $orderAddressType = null;

	public ?Order        $order        = null;
	public ?OrderAddress $orderAddress = null;

	public function __construct(
		protected Orders    $ordersService,
		protected Countries $countriesService,
	)
	{
	}

	public function render(): void
	{
		$this->template->address = $this->orderAddress;
		$this->template->render($this->getTemplateFile('OrderAddressForm', 'Order'));
	}

	#[Override]
	public function attached(IComponent $presenter): void
	{
		parent::attached($presenter);

		if ($this->orderId && $this->orderAddressType && !$this->orderAddress) {
			$this->setOrderAddress((int) $this->orderId, $this->orderAddressType);
		}
	}

	/*******************************************************************************************************************
	 * ======================== Handler
	 */


	/*******************************************************************************************************************
	 * ======================== Components
	 */

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();

		$form->addGroup('1');
		$form->addText('firstName', 'default.firstName')->setMaxLength(255);
		$form->addText('lastName', 'default.lastName')->setRequired()->setMaxLength(255);
		$form->addText('street', 'default.street')->setMaxLength(255);
		$form->addText('city', 'default.city')->setMaxLength(255);
		$form->addText('postal', 'default.zipNumber')->setMaxLength(255);
		$countries = $this->countriesService->getAllNameColumn();
		$form->addSelect('country', 'eshopOrders.orderPage.country', $countries);

		$form->addGroup('2');
		$form->addText('email', 'default.email')->setMaxLength(255);
		$form->addText('phone', 'default.phone')->setMaxLength(255);
		$form->addText('company', 'eshopOrders.orderPage.company')->setMaxLength(255);
		$form->addText('idNumber', 'eshopOrders.orderPage.idNumber')->setMaxLength(255);
		$form->addText('vatNumber', 'eshopOrders.orderPage.vatNumber')->setMaxLength(255);

		if (EshopOrdersConfig::load('customer.showIdVatNumber', false)) {
			$form->addText('idVatNumber', 'eshopOrders.orderPage.idVatNumber')->setMaxLength(255);
		}

		$form->addSaveCancelControl();
		$form->onSuccess[] = $this->formSuccess(...);

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			if ($this->orderAddress instanceof OrderAddress) {
				$orderAddress = $this->orderAddress;
				$flashMessage = 'eshopOrders.addressForm.edited';
			} else {
				throw new InvalidArgumentException("order address must be set");
			}

			$orderAddress->setFirstName($values->firstName);
			$orderAddress->setLastName($values->lastName);
			$orderAddress->setStreet($values->street);
			$orderAddress->setCity($values->city);
			$orderAddress->setPostal($values->postal);
			$orderAddress->setEmail($values->email);
			$orderAddress->setPhone($values->phone);
			$orderAddress->setCompany($values->company);
			$orderAddress->setIdNumber($values->idNumber);
			$orderAddress->setVatNumber($values->vatNumber);

			if (EshopOrdersConfig::load('customer.showIdVatNumber', false)) {
				$orderAddress->idVatNumber = $values->idVatNumber;
			}

			if ($values->country) {
				$orderAddress->setCountry($this->countriesService->getReference($values->country));
			}

			$this->em->persist($orderAddress);

			if (!$orderAddress->getId()) {
				if ($this->orderAddressType == 'delivery') {
					$this->order->setAddressDelivery($orderAddress);
				} else {
					$this->order->setAddressInvoice($orderAddress);
				}

				$this->em->persist($this->order);
			}

			$this->em->flush();

			$this->presenter->flashMessageSuccess($flashMessage);
		} catch (Exception $e) {
			$form->addError($e->getMessage());
			$this->redrawControl('form');

			return false;
		}

		return true;
	}

	public function setOrderAddress(int $orderId, string $addressType = 'delivery'): void
	{
		$this->order            = $this->ordersService->get($orderId);
		$this->orderId          = $orderId;
		$this->orderAddressType = $addressType;
		if ($addressType === 'delivery') {
			$this->orderAddress = $this->order->getAddressDeliveryRaw();
			if (!$this->orderAddress && $this->order->getAddressInvoiceRaw()) {
				$this->orderAddress = clone $this->order->getAddressInvoiceRaw();
			}
		} else {
			$this->orderAddress = $this->order->getAddressInvoiceRaw();

			$deliveryAddress = $this->order->getAddressDeliveryRaw();
			if (!$deliveryAddress) {
				$deliveryAddress = clone $this->orderAddress;
				$this->em->persist($deliveryAddress);
				$this->order->setAddressDelivery($deliveryAddress);

				$this->em->persist($this->order);
			}
		}

		if ($this->orderAddress) {
			$i = $this->orderAddress;
			$this['form']->setDefaults([
				'firstName'   => $i->getFirstName(),
				'lastName'    => $i->getLastName(),
				'street'      => $i->getStreet(),
				'city'        => $i->getCity(),
				'postal'      => $i->getPostal(),
				'country'     => $i->getCountry() ? $i->getCountry()->getId() : null,
				'email'       => $i->getEmail(),
				'phone'       => $i->getPhone(),
				'company'     => $i->getCompany(),
				'idNumber'    => $i->getIdNumber(),
				'vatNumber'   => $i->getVatNumber(),
				'idVatNumber' => $i->idVatNumber,
			]);
		}
	}
}
