<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Order;

use Core\Model\Event\CreateFormEvent;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\Event\SetFormDataEvent;
use Core\Model\Helpers\Strings;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopOrders\AdminModule\Model\Speditions;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderSpedition;
use EshopOrders\Model\Entities\Spedition;
use EshopOrders\Model\Orders;
use Exception;
use Nette\Application\Attributes\Persistent;
use Nette\Application\UI\Presenter;
use Nette\Utils\ArrayHash;

class OrderSpeditionForm extends BaseControl
{
	#[Persistent]
	public ?int $orderSpeditionId = null;

	public ?OrderSpedition $orderSpedition = null;
	public ?Order          $order          = null;

	protected ?array $cSped = null;

	public function __construct(
		protected Orders     $ordersService,
		protected Speditions $speditionsService,
	)
	{
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ======================== Components
	 */

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->getElementPrototype()->setAttribute('data-allow-toggle', 'true');

		$speditionList     = [];
		$othersCountryName = $this->t('admin.others');
		foreach ($this->getSpeditions() as $v) {
			$country = $v->country ? $v->country->getText()->name : $othersCountryName;

			if (!isset($speditionList[$country])) {
				$speditionList[$country] = [];
			}

			$speditionList[$country][$v->getId()] = $v->getName() . ' (cena ' . $v->getPrice() . ' Kč)';
		}

		if (isset($speditionList[$othersCountryName])) {
			$tmp = $speditionList[$othersCountryName];
			unset($speditionList[$othersCountryName]);

			$speditionList[$othersCountryName] = $tmp;
			$tmp                               = null;
		}

		$form->addSelect('spedition', 'eshopOrders.default.spedition', $speditionList);
		$form->addText('price', 'eshopOrders.default.speditionPrice')
			->setHtmlType('number')
			->setHtmlAttribute('step', .01)
			->setHtmlAttribute('min', 0);
		$form->addText('vat', 'eshopOrders.spedition.vatRate')
			->setHtmlType('number')
			->setHtmlAttribute('step', 1)
			->setHtmlAttribute('min', 0);

		$event          = new CreateFormEvent(
			$form,
			$this->getPresenter(false) instanceof Presenter ? $this->template : null);
		$event->control = $this;

		$this->eventDispatcher->dispatch($event, OrderSpeditionForm::class . '::createForm');

		$form->addSaveCancelControl();
		$form->onSuccess[] = $this->formSuccess(...);

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			$order          = $this->order;
			$orderSpedition = $this->order->getSpedition();
			$newSpedition   = $this->speditionsService->get($values->spedition);

			if (!$orderSpedition || !$newSpedition) {
				return false;
			}

			$orderSpedition->setPrice((float) Strings::formatEntityDecimal($values->price));
			$orderSpedition->setVatRate((int) $values->vat);

			if (!$orderSpedition->getSpedition() || $orderSpedition->getSpedition()->getId() != $newSpedition->getId()) {
				$orderSpedition->setName($newSpedition->getName());
				$orderSpedition->setSpedition($newSpedition);
			}

			$this->em->persist($orderSpedition);

			$event                   = new FormSuccessEvent(
				$form,
				$values,
				$this->presenter ? $this->template : null, $this->presenter ?: null,
			);
			$event->custom['entity'] = $orderSpedition;
			$event->control          = $this;
			$this->eventDispatcher->dispatch($event, OrderSpeditionForm::class . '::formSuccess');

			$this->em->flush();

			$this->presenter->flashMessageSuccess('eshopOrders.orderSpeditionForm.edited');
		} catch (Exception $e) {
			$form->addError($e->getMessage());
			$this->redrawControl('form');

			return false;
		}

		return true;
	}

	public function setOrder(int $orderId): void
	{
		$this->order          = $this->ordersService->get($orderId);
		$this->orderSpedition = $this->order->getSpedition();

		if ($this->orderSpedition) {
			$i = $this->orderSpedition;

			$defaults = [
				'price' => (string) str_replace(',', '.', (string) $i->getPrice()),
				'vat'   => $i->getVatRate(),
			];

			if ($i->getSpedition()) {
				$defaults['spedition'] = $i->getSpedition()->getId();
			}

			$this['form']->setDefaults($defaults);

			$this->eventDispatcher->dispatch(new SetFormDataEvent($this['form'], $this->orderSpedition), OrderSpeditionForm::class . '::setOrder');
		}
	}

	/**
	 * @return Spedition[]
	 */
	public function getSpeditions(): array
	{
		if ($this->cSped === null) {
			$this->cSped = $this->speditionsService->getAll() ?: [];
		}

		return $this->cSped;
	}
}
